import { defineComponent, ref, provide, h as h$1, Transition, withDirectives, inject, watch, reactive, onBeforeMount, onMounted, onUpdated, toRefs, onUnmounted, cloneVNode, vShow, Teleport, onBeforeUnmount, computed } from 'vue';

const CAccordion = defineComponent({
    name: 'CAccordion',
    props: {
        /**
         * The active item key.
         */
        activeItemKey: {
            type: [Number, String],
            default: undefined,
            require: false,
        },
        /**
         * Make accordion items stay open when another item is opened
         */
        alwaysOpen: Boolean,
        /**
         * Removes the default background-color, some borders, and some rounded corners to render accordions edge-to-edge with their parent container.
         */
        flush: Boolean,
    },
    setup(props, { slots }) {
        const activeItemKey = ref(props.activeItemKey);
        const setActiveItemKey = (key) => {
            activeItemKey.value = key;
        };
        provide('activeItemKey', activeItemKey);
        provide('alwaysOpen', props.alwaysOpen);
        provide('setActiveItemKey', setActiveItemKey);
        return () => h$1('div', { class: ['accordion', { ['accordion-flush']: props.flush }] }, slots.default && slots.default());
    },
});

const vVisible = {
    beforeMount(el, { value }, { transition }) {
        el._vod = el.style.display === 'none' ? '' : el.style.display;
        if (transition && value) {
            transition.beforeEnter(el);
        }
    },
    mounted(el, { value }, { transition }) {
        if (transition && value) {
            transition.enter(el);
        }
    },
    updated(el, { value, oldValue }, { transition }) {
        if (!value === !oldValue)
            return;
        if (transition) {
            if (value) {
                transition.beforeEnter(el);
                transition.enter(el);
            }
            else {
                transition.leave(el, () => {
                    // setDisplay(el, false)
                });
            }
        }
    },
};

const execute = (callback) => {
    if (typeof callback === 'function') {
        callback();
    }
};
const executeAfterTransition = (callback, transitionElement, waitForTransition = true) => {
    if (!waitForTransition) {
        execute(callback);
        return;
    }
    const durationPadding = 5;
    const emulatedDuration = getTransitionDurationFromElement(transitionElement) + durationPadding;
    let called = false;
    const handler = ({ target }) => {
        if (target !== transitionElement) {
            return;
        }
        called = true;
        transitionElement.removeEventListener('transitionend', handler);
        execute(callback);
    };
    transitionElement.addEventListener('transitionend', handler);
    setTimeout(() => {
        if (!called) {
            triggerTransitionEnd(transitionElement);
        }
    }, emulatedDuration);
};
const getTransitionDurationFromElement = (element) => {
    if (!element) {
        return 0;
    }
    // Get transition-duration of the element
    let { transitionDuration, transitionDelay } = window.getComputedStyle(element);
    const floatTransitionDuration = Number.parseFloat(transitionDuration);
    const floatTransitionDelay = Number.parseFloat(transitionDelay);
    // Return 0 if element or transition duration is not found
    if (!floatTransitionDuration && !floatTransitionDelay) {
        return 0;
    }
    // If multiple durations are defined, take the first
    transitionDuration = transitionDuration.split(',')[0];
    transitionDelay = transitionDelay.split(',')[0];
    return (Number.parseFloat(transitionDuration) + Number.parseFloat(transitionDelay)) * 1000;
};
const triggerTransitionEnd = (element) => {
    element.dispatchEvent(new Event('transitionend'));
};

const CCollapse = defineComponent({
    name: 'CCollapse',
    props: {
        /**
         * Set horizontal collapsing to transition the width instead of height.
         */
        horizontal: {
            type: Boolean,
            required: false,
        },
        /**
         * Toggle the visibility of component.
         */
        visible: {
            type: Boolean,
            required: false,
        },
    },
    emits: [
        /**
         * Callback fired when the component requests to be hidden.
         */
        'hide',
        /**
         * Callback fired when the component requests to be shown.
         */
        'show',
    ],
    setup(props, { slots, emit }) {
        const collapsing = ref(false);
        const show = ref(props.visible);
        const handleBeforeEnter = () => {
            collapsing.value = true;
        };
        const handleEnter = (el, done) => {
            emit('show');
            // collapsing.value = true
            setTimeout(() => {
                executeAfterTransition(() => done(), el);
                if (props.horizontal) {
                    el.style.width = `${el.scrollWidth}px`;
                    return;
                }
                el.style.height = `${el.scrollHeight}px`;
            }, 1);
        };
        const handleAfterEnter = (el) => {
            show.value = true;
            collapsing.value = false;
            props.horizontal ? el.style.removeProperty('width') : el.style.removeProperty('height');
        };
        const handleBeforeLeave = (el) => {
            collapsing.value = true;
            show.value = false;
            if (props.horizontal) {
                el.style.width = `${el.scrollWidth}px`;
                return;
            }
            el.style.height = `${el.scrollHeight}px`;
        };
        const handleLeave = (el, done) => {
            emit('hide');
            setTimeout(() => {
                executeAfterTransition(() => done(), el);
                if (props.horizontal) {
                    el.style.width = '0px';
                    return;
                }
                el.style.height = '0px';
            }, 1);
        };
        const handleAfterLeave = (el) => {
            collapsing.value = false;
            props.horizontal ? el.style.removeProperty('width') : el.style.removeProperty('height');
        };
        return () => h$1(Transition, {
            css: false,
            onBeforeEnter: () => handleBeforeEnter(),
            onEnter: (el, done) => handleEnter(el, done),
            onAfterEnter: (el) => handleAfterEnter(el),
            onBeforeLeave: (el) => handleBeforeLeave(el),
            onLeave: (el, done) => handleLeave(el, done),
            onAfterLeave: (el) => handleAfterLeave(el),
        }, () => withDirectives(h$1('div', {
            class: [
                collapsing.value ? 'collapsing' : 'collapse',
                { 'collapse-horizontal': props.horizontal, show: show.value },
            ],
        }, slots.default && slots.default()), [[vVisible, props.visible]]));
    },
});

const CAccordionBody = defineComponent({
    name: 'CAccordionBody',
    setup(_, { slots }) {
        const visible = inject('visible');
        return () => h$1(CCollapse, { class: 'accordion-collapse', visible: visible.value }, {
            default: () => h$1('div', { class: ['accordion-body'] }, slots.default && slots.default()),
        });
    },
});

const CAccordionButton = defineComponent({
    name: 'CAccordionButton',
    setup(_, { slots }) {
        const toggleVisibility = inject('toggleVisibility');
        const visible = inject('visible');
        return () => h$1('button', {
            type: 'button',
            'aria-expanded': !visible.value,
            class: ['accordion-button', { ['collapsed']: !visible.value }],
            onClick: () => toggleVisibility(),
        }, slots.default && slots.default());
    },
});

const CAccordionHeader = defineComponent({
    name: 'CAccordionHeader',
    setup(_, { slots }) {
        return () => h$1('div', { class: ['accordion-header'] }, h$1(CAccordionButton, {}, {
            default: () => slots.default && slots.default(),
        }));
    },
});

const CAccordionItem = defineComponent({
    name: 'CAccordionItem',
    props: {
        /**
         * The item key.
         */
        itemKey: {
            type: [Number, String],
            default: undefined,
            require: false,
        },
    },
    setup(props, { slots }) {
        const activeItemKey = inject('activeItemKey');
        const alwaysOpen = inject('alwaysOpen');
        // eslint-disable-next-line no-unused-vars
        const setActiveItemKey = inject('setActiveItemKey');
        const itemKey = ref(props.itemKey ? props.itemKey : Math.random().toString(36).substr(2, 9));
        const visible = ref(Boolean(activeItemKey.value === itemKey.value));
        watch(activeItemKey, () => (visible.value = Boolean(activeItemKey.value === itemKey.value)));
        const toggleVisibility = () => {
            visible.value = !visible.value;
            !alwaysOpen && visible && setActiveItemKey(itemKey.value);
        };
        provide('visible', visible);
        provide('toggleVisibility', toggleVisibility);
        return () => h$1('div', { class: ['accordion-item'] }, slots.default && slots.default());
    },
});

const CAccordionPlugin = {
    install: (app) => {
        app.component(CAccordion.name, CAccordion);
        app.component(CAccordionBody.name, CAccordionBody);
        app.component(CAccordionButton.name, CAccordionButton);
        app.component(CAccordionHeader.name, CAccordionHeader);
        app.component(CAccordionItem.name, CAccordionItem);
    },
};

const Shape = {
    type: String,
    validator: (value) => {
        // The value must match one of these strings
        return [
            'rounded',
            'rounded-top',
            'rounded-end',
            'rounded-bottom',
            'rounded-start',
            'rounded-circle',
            'rounded-pill',
            'rounded-0',
            'rounded-1',
            'rounded-2',
            'rounded-3',
        ].includes(value);
    },
};
const Color = {
    type: String,
    validator: (value) => {
        // The value must match one of these strings
        return [
            'primary',
            'secondary',
            'success',
            'danger',
            'warning',
            'info',
            'dark',
            'light',
            'link',
            'transparent',
        ].includes(value);
    },
};
const TextColor = {
    type: String,
    validator: (value) => {
        return [
            'primary',
            'secondary',
            'success',
            'danger',
            'warning',
            'info',
            'dark',
            'light',
            'white',
            'muted',
            'high-emphasis',
            'medium-emphasis',
            'disabled',
            'high-emphasis-inverse',
            'medium-emphasis-inverse',
            'disabled-inverse',
        ].includes(value);
    },
};

const CCloseButton = defineComponent({
    name: 'CCloseButton',
    props: {
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * Change the default color to white.
         */
        white: {
            type: Boolean,
            required: false,
        },
    },
    emits: [
        /**
         * Event called when the user clicks on the component.
         */
        'click',
    ],
    setup(props, { emit }) {
        const handleClick = () => {
            if (props.disabled) {
                return;
            }
            emit('click');
        };
        return () => h$1('button', {
            type: 'button',
            class: [
                'btn',
                'btn-close',
                {
                    ['btn-close-white']: props.white,
                },
                props.disabled,
            ],
            'aria-label': 'Close',
            disabled: props.disabled,
            onClick: handleClick,
        });
    },
});

const CAlert = defineComponent({
    name: 'CAlert',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Optionally add a close button to alert and allow it to self dismisss.
         */
        dismissible: {
            type: Boolean,
            default: false,
            required: false,
        },
        /**
         * Set the alert variant to a solid.
         *
         * @values 'solid'
         */
        variant: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return value === 'solid';
            },
        },
        /**
         * Toggle the visibility of alert component.
         */
        visible: {
            type: Boolean,
            default: true,
            required: false,
        },
    },
    emits: [
        /**
         * Callback fired when the component requests to be closed.
         */
        'close',
    ],
    setup(props, { slots, emit }) {
        const visible = ref(props.visible);
        watch(() => props.visible, () => {
            visible.value = props.visible;
        });
        const handleDismiss = () => {
            visible.value = false;
            emit('close');
        };
        return () => h$1(Transition, {
            name: 'fade',
            duration: 350,
            enterFromClass: '',
            enterActiveClass: 'fade',
            enterToClass: 'fade show',
            leaveActiveClass: 'fade',
        }, {
            default: () => visible.value &&
                h$1('div', {
                    class: [
                        'alert',
                        props.variant === 'solid'
                            ? `bg-${props.color} text-white border-0`
                            : `alert-${props.color}`,
                        {
                            [`alert-${props.color}`]: props.color,
                            'alert-dismissible': props.dismissible,
                        },
                    ],
                }, [
                    slots.default && slots.default(),
                    props.dismissible &&
                        h$1(CCloseButton, {
                            onClick: () => {
                                handleDismiss();
                            },
                        }),
                ]),
        });
    },
});

const CAlertHeading = defineComponent({
    name: 'CAlertHeading',
    props: {
        /**
         * 	Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'h4',
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: 'alert-heading',
        }, slots);
    },
});

const CAlertLink = defineComponent({
    name: 'CAlertLink',
    setup(_, { slots }) {
        return () => h$1('a', {
            class: 'alert-link',
        }, slots);
    },
});

const CAlertPlugin = {
    install: (app) => {
        app.component(CAlert.name, CAlert);
        app.component(CAlertHeading.name, CAlertHeading);
        app.component(CAlertLink.name, CAlertLink);
    },
};

const CAvatar = defineComponent({
    name: 'CAvatar',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Select the shape of the component.
         *
         * @values 'rounded', 'rounded-top', 'rounded-end', 'rounded-bottom', 'rounded-start', 'rounded-circle', 'rounded-pill', 'rounded-0', 'rounded-1', 'rounded-2', 'rounded-3'
         */
        shape: Shape,
        /**
         * Size the component small, large, or extra large.
         *
         * @values 'sm', 'md', 'lg', 'xl'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['sm', 'md', 'lg', 'xl'].includes(value);
            },
        },
        /**
         * The src attribute for the img element.
         */
        src: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Sets the color context of the status indicator to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        status: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return [
                    'primary',
                    'secondary',
                    'success',
                    'danger',
                    'warning',
                    'info',
                    'dark',
                    'light',
                ].includes(value);
            },
        },
        /**
         * Sets the text color of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', 'white', 'muted', 'high-emphasis', 'medium-emphasis', 'disabled', 'high-emphasis-inverse', 'medium-emphasis-inverse', 'disabled-inverse'
         */
        textColor: TextColor,
    },
    setup(props, { slots }) {
        return () => h$1('div', {
            class: [
                'avatar',
                {
                    [`bg-${props.color}`]: props.color,
                    [`avatar-${props.size}`]: props.size,
                    [`text-${props.textColor}`]: props.textColor,
                },
                `${props.shape}`,
            ],
        }, [
            props.src
                ? h$1('img', { src: props.src, class: 'avatar-img' })
                : slots.default && slots.default(),
            props.status && h$1('span', { class: ['avatar-status', `bg-${props.status}`] }),
        ]);
    },
});

const CAvatarPlugin = {
    install: (app) => {
        app.component(CAvatar.name, CAvatar);
    },
};

const CBackdrop = defineComponent({
    name: 'CBackdrop',
    props: {
        /**
         * Toggle the visibility of modal component.
         */
        visible: {
            type: Boolean,
            default: false,
            required: false,
        },
    },
    setup(props) {
        const handleBeforeEnter = (el) => {
            el.classList.remove('d-none');
        };
        const handleEnter = (el, done) => {
            executeAfterTransition(() => done(), el);
            setTimeout(() => {
                el.style.visibility = 'visible';
                el.classList.add('show');
            }, 1);
        };
        const handleLeave = (el, done) => {
            executeAfterTransition(() => done(), el);
            el.classList.remove('show');
        };
        const handleAfterLeave = (el) => {
            el.classList.add('d-none');
        };
        return () => h$1(Transition, {
            onBeforeEnter: (el) => handleBeforeEnter(el),
            onEnter: (el, done) => handleEnter(el, done),
            onLeave: (el, done) => handleLeave(el, done),
            onAfterLeave: (el) => handleAfterLeave(el),
        }, () => props.visible &&
            h$1('div', {
                class: 'fade',
            }));
    },
});

const CBackdropPlugin = {
    install: (app) => {
        app.component(CBackdrop.name, CBackdrop);
    },
};

const CBadge = defineComponent({
    name: 'CBadge',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'span',
        },
        /**
         * Position badge in one of the corners of a link or button.
         *
         * @values 'top-start', 'top-end', 'bottom-end', 'bottom-start'
         */
        position: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['top-start', 'top-end', 'bottom-end', 'bottom-start'].includes(value);
            },
        },
        /**
         * Select the shape of the component.
         *
         * @values 'rounded', 'rounded-top', 'rounded-end', 'rounded-bottom', 'rounded-start', 'rounded-circle', 'rounded-pill', 'rounded-0', 'rounded-1', 'rounded-2', 'rounded-3'
         */
        shape: Shape,
        /**
         * Size the component small.
         *
         * @values 'sm'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return value === 'sm';
            },
        },
        /**
         * Sets the text color of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', 'white', 'muted', 'high-emphasis', 'medium-emphasis', 'disabled', 'high-emphasis-inverse', 'medium-emphasis-inverse', 'disabled-inverse'
         */
        textColor: TextColor,
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: [
                'badge',
                {
                    [`bg-${props.color}`]: props.color,
                    'position-absolute translate-middle': props.position,
                    'top-0': props.position && props.position.includes('top'),
                    'top-100': props.position && props.position.includes('bottom'),
                    'start-100': props.position && props.position.includes('end'),
                    'start-0': props.position && props.position.includes('start'),
                    [`badge-${props.size}`]: props.size,
                    [`text-${props.textColor}`]: props.textColor,
                },
                props.shape,
            ],
        }, slots.default && slots.default());
    },
});

const CBadgePlugin = {
    install: (app) => {
        app.component(CBadge.name, CBadge);
    },
};

const CBreadcrumbItem = defineComponent({
    name: 'CBreadcrumbItem',
    props: {
        /**
         * Toggle the active state for the component.
         */
        active: {
            type: Boolean,
            required: false,
        },
        /**
         * The `href` attribute for the inner link component.
         */
        href: {
            type: String,
            default: undefined,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1('li', {
            class: [
                'breadcrumb-item',
                {
                    active: props.active,
                },
            ],
            ...(props.active && { 'aria-current': 'page' }),
        }, props.href
            ? h$1('a', { href: props.href }, slots.default && slots.default())
            : slots.default && slots.default());
    },
});

const CBreadcrumb = defineComponent({
    name: 'CBreadcrumb',
    setup(_, { slots, attrs }) {
        return () => h$1('nav', {
            'aria-label': 'breadcrumb',
        }, h$1('ol', { class: ['breadcrumb', attrs.class] }, slots.default && slots.default()));
    },
});

const CBreadcrumbPlugin = {
    install: (app) => {
        app.component(CBreadcrumb.name, CBreadcrumb);
        app.component(CBreadcrumbItem.name, CBreadcrumbItem);
    },
};

const CButton = defineComponent({
    name: 'CButton',
    props: {
        /**
         * Toggle the active state for the component.
         */
        active: {
            type: Boolean,
            default: false,
            required: false,
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'button',
            required: false,
        },
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * The href attribute specifies the URL of the page the link goes to.
         */
        href: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Select the shape of the component.
         *
         * @values 'rounded', 'rounded-top', 'rounded-end', 'rounded-bottom', 'rounded-start', 'rounded-circle', 'rounded-pill', 'rounded-0', 'rounded-1', 'rounded-2', 'rounded-3'
         */
        shape: Shape,
        /**
         * Size the component small or large.
         *
         * @values 'sm', 'lg'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['sm', 'lg'].includes(value);
            },
        },
        /**
         * Specifies the type of button. Always specify the type attribute for the `<button>` element.
         * Different browsers may use different default types for the `<button>` element.
         *
         * @values 'button', 'submit', 'reset'
         */
        type: {
            type: String,
            default: 'button',
            required: false,
            validator: (value) => {
                return ['button', 'submit', 'reset'].includes(value);
            },
        },
        /**
         * Set the button variant to an outlined button or a ghost button.
         *
         * @values 'ghost', 'outline'
         */
        variant: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['ghost', 'outline'].includes(value);
            },
        },
    },
    emits: [
        /**
         * Event called when the user clicks on the button.
         */
        'click',
    ],
    setup(props, { emit, slots }) {
        const handleClick = (event) => {
            if (props.disabled) {
                return;
            }
            emit('click', event);
        };
        return () => h$1(props.component, {
            class: [
                'btn',
                props.variant ? `btn-${props.variant}-${props.color}` : `btn-${props.color}`,
                {
                    [`btn-${props.size}`]: props.size,
                    active: props.active,
                    disabled: props.disabled,
                },
                props.shape,
            ],
            disabled: props.disabled && props.component !== 'a',
            ...(props.component === 'a' && props.disabled && { 'aria-disabled': true, tabIndex: -1 }),
            ...(props.component === 'a' && props.href && { href: props.href }),
            ...(props.component === 'button' && { type: props.type }),
            onClick: handleClick,
        }, slots.default && slots.default());
    },
});

const CButtonPlugin = {
    install: (app) => {
        app.component(CButton.name, CButton);
    },
};

const CButtonToolbar = defineComponent({
    name: 'CButtonToolbar',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'btn-toolbar' }, slots.default && slots.default());
    },
});

const CButtonGroup = defineComponent({
    name: 'CButtonGroup',
    props: {
        /**
         * Size the component small or large.
         *
         * @values 'sm', 'lg'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['sm', 'lg'].includes(value);
            },
        },
        /**
         * Create a set of buttons that appear vertically stacked rather than horizontally. Split button dropdowns are not supported here.
         */
        vertical: {
            type: Boolean,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1('div', {
            class: [
                props.vertical ? 'btn-group-vertical' : 'btn-group',
                { [`btn-group-${props.size}`]: props.size },
            ],
        }, slots.default && slots.default());
    },
});

const CButtonGroupPlugin = {
    install: (app) => {
        app.component(CButtonToolbar.name, CButtonToolbar);
        app.component(CButtonGroup.name, CButtonGroup);
    },
};

const CCallout = defineComponent({
    name: 'CCallout',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
    },
    setup(props, { slots }) {
        return () => h$1('div', {
            class: [
                'callout',
                {
                    [`callout-${props.color}`]: props.color,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CCalloutPlugin = {
    install: (app) => {
        app.component(CCallout.name, CCallout);
    },
};

const CCard = defineComponent({
    name: 'CCard',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Sets the text color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', 'white', 'muted', 'high-emphasis', 'medium-emphasis', 'disabled', 'high-emphasis-inverse', 'medium-emphasis-inverse', 'disabled-inverse'
         */
        textColor: TextColor,
    },
    setup(props, { slots }) {
        return () => h$1('div', {
            class: [
                'card',
                {
                    [`bg-${props.color}`]: props.color,
                    [`text-${props.textColor}`]: props.textColor,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CCardBody = defineComponent({
    name: 'CCardBody',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'card-body' }, slots.default && slots.default());
    },
});

const CCardFooter = defineComponent({
    name: 'CCardFooter',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'card-footer' }, slots.default && slots.default());
    },
});

const CCardGroup = defineComponent({
    name: 'CCardGroup',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'card-group' }, slots.default && slots.default());
    },
});

const CCardHeader = defineComponent({
    name: 'CCardHeader',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'div',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'card-header' }, slots.default && slots.default());
    },
});

const CCardImage = defineComponent({
    name: 'CCardImage',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'img',
        },
        /**
         * Optionally orientate the image to the top, bottom.
         *
         * @values 'top', 'bottom'
         */
        orientation: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['top', 'bottom'].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: `card-img${props.orientation ? `-${props.orientation}` : ''}`,
        }, slots.default && slots.default());
    },
});

const CCardImageOverlay = defineComponent({
    name: 'CCardImageOverlay',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'card-img-overlay' }, slots.default && slots.default());
    },
});

const CLink = defineComponent({
    name: 'CLink',
    props: {
        /**
         * Toggle the active state for the component.
         */
        active: {
            type: Boolean,
            required: false,
        },
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'a',
        },
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * The href attribute specifies the URL of the page the link goes to.
         */
        href: {
            type: String,
            default: undefined,
            required: false,
        },
    },
    emits: [
        /**
         * Event called when the user clicks on the component.
         */
        'click',
    ],
    setup(props, { slots, emit }) {
        const handleClick = () => {
            emit('click', props.href);
        };
        return () => h$1(props.component, {
            class: [{ active: props.active, disabled: props.disabled }],
            ...(props.active && { 'aria-current': 'page' }),
            ...(props.component === 'a' && props.disabled && { 'aria-disabled': true, tabIndex: -1 }),
            ...((props.component === 'a' || props.component === 'button') && {
                onClick: handleClick,
            }),
            href: props.href,
        }, slots.default && slots.default());
    },
});

const CCLinkPlugin = {
    install: (app) => {
        app.component(CLink.name, CLink);
    },
};

const CCardLink = defineComponent({
    name: 'CCardLink',
    props: {
        /**
         * The href attribute specifies the URL of the page the link goes to.
         */
        href: {
            type: String,
            default: '#',
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(CLink, { class: 'card-link', href: props.href }, { default: () => slots.default && slots.default() });
    },
});

const CCardSubtitle = defineComponent({
    name: 'CCardSubtitle',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'h6',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'card-subtitle' }, slots.default && slots.default());
    },
});

const CCardText = defineComponent({
    name: 'CCardText',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'p',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'card-text' }, slots.default && slots.default());
    },
});

const CCardTitle = defineComponent({
    name: 'CCardTitle',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'h5',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'card-title' }, slots.default && slots.default());
    },
});

const CCardPlugin = {
    install: (app) => {
        app.component(CCard.name, CCard);
        app.component(CCardBody.name, CCardBody);
        app.component(CCardFooter.name, CCardFooter);
        app.component(CCardGroup.name, CCardGroup);
        app.component(CCardHeader.name, CCardHeader);
        app.component(CCardImage.name, CCardImage);
        app.component(CCardImageOverlay.name, CCardImageOverlay);
        app.component(CCardLink.name, CCardLink);
        app.component(CCardSubtitle.name, CCardSubtitle);
        app.component(CCardText.name, CCardText);
        app.component(CCardTitle.name, CCardTitle);
    },
};

const isInViewport = (element) => {
    const rect = element.getBoundingClientRect();
    return (Math.floor(rect.top) >= 0 &&
        Math.floor(rect.left) >= 0 &&
        Math.floor(rect.bottom) <= (window.innerHeight || document.documentElement.clientHeight) &&
        Math.floor(rect.right) <= (window.innerWidth || document.documentElement.clientWidth));
};

const CCarousel = defineComponent({
    name: 'CCarousel',
    props: {
        /**
         * Adding in the previous and next controls.
         */
        controls: Boolean,
        /**
         * Add darker controls, indicators, and captions.
         */
        dark: Boolean,
        /**
         * index of the active item.
         */
        index: {
            type: Number,
            default: 0,
            required: false,
        },
        /**
         * Adding indicators at the bottom of the carousel for each item.
         */
        indicators: Boolean,
        /**
         * The amount of time to delay between automatically cycling an item. If false, carousel will not automatically cycle.
         */
        interval: {
            type: [Boolean, Number],
            default: 5000,
            required: false,
        },
        /**
         * If set to 'hover', pauses the cycling of the carousel on mouseenter and resumes the cycling of the carousel on mouseleave. If set to false, hovering over the carousel won't pause it.
         */
        pause: {
            type: [Boolean, String],
            default: 'hover',
            required: false,
            validator: (value) => {
                return typeof value === 'boolean' || value === 'hover';
            },
        },
        /**
         * Set type of the transition.
         *
         * @values 'crossfade', 'slide'
         */
        transition: {
            type: String,
            default: 'slide',
            required: false,
            validator: (value) => {
                return ['crossfade', 'slide'].includes(value);
            },
        },
        /**
         * Set whether the carousel should cycle continuously or have hard stops.
         */
        wrap: {
            type: Boolean,
            default: true,
            required: false,
        },
    },
    setup(props, { slots }) {
        const carouselRef = ref();
        const timeout = ref();
        const animating = ref(false);
        const visible = ref();
        const customInterval = ref(props.interval);
        const setAnimating = (value) => {
            animating.value = value;
        };
        const setCustomInterval = (value) => {
            customInterval.value = value;
        };
        provide('setAnimating', setAnimating);
        provide('setCustomInterval', setCustomInterval);
        const pause = () => timeout.value && clearInterval(timeout.value);
        const cycle = () => {
            pause();
            if (typeof props.interval === 'number') {
                timeout.value = setTimeout(() => nextItemWhenVisible(), typeof customInterval.value === 'number' ? customInterval.value : props.interval);
            }
        };
        const state = reactive({
            items: [],
            active: props.index,
            direction: 'next',
        });
        onBeforeMount(() => {
            if (slots.default) {
                // @ts-expect-error TODO: fix types
                state.items = slots.default().filter((child) => child.type.name === 'CCarouselItem');
            }
        });
        const handleControlClick = (direction) => {
            if (animating.value) {
                return;
            }
            state.direction = direction;
            if (direction === 'next') {
                state.active === state.items.length - 1 ? (state.active = 0) : state.active++;
            }
            else {
                state.active === 0 ? (state.active = state.items.length - 1) : state.active--;
            }
        };
        const nextItemWhenVisible = () => {
            // Don't call next when the page isn't visible
            // or the carousel or its parent isn't visible
            if (!document.hidden && isInViewport(carouselRef.value)) {
                handleControlClick('next');
            }
        };
        const handleIndicatorClick = (index) => {
            if (state.active === index) {
                return;
            }
            if (state.active < index) {
                state.direction = 'next';
                state.active = index;
                return;
            }
            if (state.active > index) {
                state.direction = 'prev';
                state.active = index;
            }
        };
        const handleScroll = () => {
            if (!document.hidden && isInViewport(carouselRef.value)) {
                visible.value = true;
            }
            else {
                visible.value = false;
            }
        };
        onMounted(() => {
            window.addEventListener('scroll', handleScroll);
        });
        onUpdated(() => {
            watch(animating, () => {
                if (props.wrap) {
                    !animating.value && cycle();
                    return;
                }
                if (!props.wrap && state.active < state.items.length - 1) {
                    !animating.value && cycle();
                }
            });
        });
        watch(visible, () => {
            visible.value && cycle();
        });
        return () => h$1('div', {
            class: [
                'carousel slide',
                props.transition === 'crossfade' && 'carousel-fade',
                props.dark && 'carousel-dark',
            ],
            onmouseover: () => props.pause && pause(),
            onmouseleave: () => cycle(),
            ref: carouselRef,
        }, [
            props.indicators &&
                h$1('div', {
                    class: 'carousel-indicators',
                }, state.items.map((_, index) => {
                    return h$1('button', {
                        type: 'button',
                        id: index,
                        'data-coreui-target': '',
                        ...(state.active === index && { class: 'active' }),
                        onClick: () => handleIndicatorClick(index),
                    });
                })),
            h$1('div', { class: 'carousel-inner' }, state.items.map((item, index) => {
                return h$1(item, {
                    active: state.active === index ? true : false,
                    direction: state.direction,
                });
            })),
            props.controls && [
                h$1('button', {
                    type: 'button',
                    class: 'carousel-control-prev',
                    'data-coreui-target': '',
                    onClick: () => handleControlClick('prev'),
                }, [
                    h$1('span', { class: 'carousel-control-prev-icon', ariaHidden: 'true' }),
                    h$1('span', { class: 'visually-hidden' }, 'Previous'),
                ]),
                h$1('button', {
                    type: 'button',
                    class: 'carousel-control-next',
                    'data-coreui-target': '',
                    onClick: () => handleControlClick('next'),
                }, [
                    h$1('span', { class: 'carousel-control-next-icon', ariaHidden: 'true' }),
                    h$1('span', { class: 'visually-hidden' }, 'Next'),
                ]),
            ],
        ]);
    },
});

const CCarouselCaption = defineComponent({
    name: 'CCarouselCaption',
    setup(_, { slots }) {
        return () => h$1('div', {
            class: 'carousel-caption',
        }, slots.default && slots.default());
    },
});

const CCarouselItem = defineComponent({
    name: 'CCarouselItem',
    props: {
        /**
         * @ignore
         */
        active: {
            type: Boolean,
            default: false,
        },
        /**
         * @ignore
         */
        direction: {
            type: String,
            default: 'next',
            required: false,
        },
        /**
         * The amount of time to delay between automatically cycling an item.
         */
        interval: {
            type: [Boolean, Number],
            default: false,
            required: false,
        },
    },
    setup(props, { slots }) {
        const carouselItemRef = ref();
        const { active } = toRefs(props);
        const directionClassName = ref();
        const orderClassName = ref();
        const activeClassName = ref(active.value && 'active');
        // eslint-disable-next-line no-unused-vars
        const setAnimating = inject('setAnimating');
        // eslint-disable-next-line no-unused-vars
        const setCustomInterval = inject('setCustomInterval');
        watch(active, (active, prevActive) => {
            active && setCustomInterval(props.interval);
            if (!prevActive && active) {
                orderClassName.value = `carousel-item-${props.direction}`;
                setCustomInterval(props.interval);
            }
            setTimeout(() => {
                if (prevActive && !active) {
                    activeClassName.value = 'active';
                }
                directionClassName.value = `carousel-item-${props.direction === 'next' ? 'start' : 'end'}`;
            }, 0);
            carouselItemRef.value.addEventListener('transitionstart', () => {
                setAnimating(true);
            });
            carouselItemRef.value.addEventListener('transitionend', () => {
                setAnimating(false);
                if (active) {
                    directionClassName.value = '';
                    orderClassName.value = '';
                    activeClassName.value = 'active';
                }
                if (!active) {
                    directionClassName.value = '';
                    orderClassName.value = '';
                    activeClassName.value = '';
                }
            });
        });
        return () => h$1('div', {
            class: [
                'carousel-item',
                activeClassName.value,
                directionClassName.value,
                orderClassName.value,
            ],
            ref: carouselItemRef,
        }, slots.default && slots.default());
    },
});

const CCarouselPlugin = {
    install: (app) => {
        app.component(CCarousel.name, CCarousel);
        app.component(CCarouselCaption.name, CCarouselCaption);
        app.component(CCarouselItem.name, CCarouselItem);
    },
};

const CCloseButtonPlugin = {
    install: (app) => {
        app.component(CCloseButton.name, CCloseButton);
    },
};

const CCollapsePlugin = {
    install: (app) => {
        app.component(CCollapse.name, CCollapse);
    },
};

var top = 'top';
var bottom = 'bottom';
var right = 'right';
var left = 'left';
var auto = 'auto';
var basePlacements = [top, bottom, right, left];
var start = 'start';
var end = 'end';
var clippingParents = 'clippingParents';
var viewport = 'viewport';
var popper = 'popper';
var reference = 'reference';
var variationPlacements = /*#__PURE__*/basePlacements.reduce(function (acc, placement) {
  return acc.concat([placement + "-" + start, placement + "-" + end]);
}, []);
var placements = /*#__PURE__*/[].concat(basePlacements, [auto]).reduce(function (acc, placement) {
  return acc.concat([placement, placement + "-" + start, placement + "-" + end]);
}, []); // modifiers that need to read the DOM

var beforeRead = 'beforeRead';
var read = 'read';
var afterRead = 'afterRead'; // pure-logic modifiers

var beforeMain = 'beforeMain';
var main = 'main';
var afterMain = 'afterMain'; // modifier with the purpose to write to the DOM (or write into a framework state)

var beforeWrite = 'beforeWrite';
var write = 'write';
var afterWrite = 'afterWrite';
var modifierPhases = [beforeRead, read, afterRead, beforeMain, main, afterMain, beforeWrite, write, afterWrite];

function getNodeName(element) {
  return element ? (element.nodeName || '').toLowerCase() : null;
}

function getWindow(node) {
  if (node == null) {
    return window;
  }

  if (node.toString() !== '[object Window]') {
    var ownerDocument = node.ownerDocument;
    return ownerDocument ? ownerDocument.defaultView || window : window;
  }

  return node;
}

function isElement(node) {
  var OwnElement = getWindow(node).Element;
  return node instanceof OwnElement || node instanceof Element;
}

function isHTMLElement(node) {
  var OwnElement = getWindow(node).HTMLElement;
  return node instanceof OwnElement || node instanceof HTMLElement;
}

function isShadowRoot(node) {
  // IE 11 has no ShadowRoot
  if (typeof ShadowRoot === 'undefined') {
    return false;
  }

  var OwnElement = getWindow(node).ShadowRoot;
  return node instanceof OwnElement || node instanceof ShadowRoot;
}

// and applies them to the HTMLElements such as popper and arrow

function applyStyles(_ref) {
  var state = _ref.state;
  Object.keys(state.elements).forEach(function (name) {
    var style = state.styles[name] || {};
    var attributes = state.attributes[name] || {};
    var element = state.elements[name]; // arrow is optional + virtual elements

    if (!isHTMLElement(element) || !getNodeName(element)) {
      return;
    } // Flow doesn't support to extend this property, but it's the most
    // effective way to apply styles to an HTMLElement
    // $FlowFixMe[cannot-write]


    Object.assign(element.style, style);
    Object.keys(attributes).forEach(function (name) {
      var value = attributes[name];

      if (value === false) {
        element.removeAttribute(name);
      } else {
        element.setAttribute(name, value === true ? '' : value);
      }
    });
  });
}

function effect$2(_ref2) {
  var state = _ref2.state;
  var initialStyles = {
    popper: {
      position: state.options.strategy,
      left: '0',
      top: '0',
      margin: '0'
    },
    arrow: {
      position: 'absolute'
    },
    reference: {}
  };
  Object.assign(state.elements.popper.style, initialStyles.popper);
  state.styles = initialStyles;

  if (state.elements.arrow) {
    Object.assign(state.elements.arrow.style, initialStyles.arrow);
  }

  return function () {
    Object.keys(state.elements).forEach(function (name) {
      var element = state.elements[name];
      var attributes = state.attributes[name] || {};
      var styleProperties = Object.keys(state.styles.hasOwnProperty(name) ? state.styles[name] : initialStyles[name]); // Set all values to an empty string to unset them

      var style = styleProperties.reduce(function (style, property) {
        style[property] = '';
        return style;
      }, {}); // arrow is optional + virtual elements

      if (!isHTMLElement(element) || !getNodeName(element)) {
        return;
      }

      Object.assign(element.style, style);
      Object.keys(attributes).forEach(function (attribute) {
        element.removeAttribute(attribute);
      });
    });
  };
} // eslint-disable-next-line import/no-unused-modules


var applyStyles$1 = {
  name: 'applyStyles',
  enabled: true,
  phase: 'write',
  fn: applyStyles,
  effect: effect$2,
  requires: ['computeStyles']
};

function getBasePlacement(placement) {
  return placement.split('-')[0];
}

var max = Math.max;
var min = Math.min;
var round = Math.round;

function getUAString() {
  var uaData = navigator.userAgentData;

  if (uaData != null && uaData.brands) {
    return uaData.brands.map(function (item) {
      return item.brand + "/" + item.version;
    }).join(' ');
  }

  return navigator.userAgent;
}

function isLayoutViewport() {
  return !/^((?!chrome|android).)*safari/i.test(getUAString());
}

function getBoundingClientRect(element, includeScale, isFixedStrategy) {
  if (includeScale === void 0) {
    includeScale = false;
  }

  if (isFixedStrategy === void 0) {
    isFixedStrategy = false;
  }

  var clientRect = element.getBoundingClientRect();
  var scaleX = 1;
  var scaleY = 1;

  if (includeScale && isHTMLElement(element)) {
    scaleX = element.offsetWidth > 0 ? round(clientRect.width) / element.offsetWidth || 1 : 1;
    scaleY = element.offsetHeight > 0 ? round(clientRect.height) / element.offsetHeight || 1 : 1;
  }

  var _ref = isElement(element) ? getWindow(element) : window,
      visualViewport = _ref.visualViewport;

  var addVisualOffsets = !isLayoutViewport() && isFixedStrategy;
  var x = (clientRect.left + (addVisualOffsets && visualViewport ? visualViewport.offsetLeft : 0)) / scaleX;
  var y = (clientRect.top + (addVisualOffsets && visualViewport ? visualViewport.offsetTop : 0)) / scaleY;
  var width = clientRect.width / scaleX;
  var height = clientRect.height / scaleY;
  return {
    width: width,
    height: height,
    top: y,
    right: x + width,
    bottom: y + height,
    left: x,
    x: x,
    y: y
  };
}

// means it doesn't take into account transforms.

function getLayoutRect(element) {
  var clientRect = getBoundingClientRect(element); // Use the clientRect sizes if it's not been transformed.
  // Fixes https://github.com/popperjs/popper-core/issues/1223

  var width = element.offsetWidth;
  var height = element.offsetHeight;

  if (Math.abs(clientRect.width - width) <= 1) {
    width = clientRect.width;
  }

  if (Math.abs(clientRect.height - height) <= 1) {
    height = clientRect.height;
  }

  return {
    x: element.offsetLeft,
    y: element.offsetTop,
    width: width,
    height: height
  };
}

function contains(parent, child) {
  var rootNode = child.getRootNode && child.getRootNode(); // First, attempt with faster native method

  if (parent.contains(child)) {
    return true;
  } // then fallback to custom implementation with Shadow DOM support
  else if (rootNode && isShadowRoot(rootNode)) {
      var next = child;

      do {
        if (next && parent.isSameNode(next)) {
          return true;
        } // $FlowFixMe[prop-missing]: need a better way to handle this...


        next = next.parentNode || next.host;
      } while (next);
    } // Give up, the result is false


  return false;
}

function getComputedStyle$1(element) {
  return getWindow(element).getComputedStyle(element);
}

function isTableElement(element) {
  return ['table', 'td', 'th'].indexOf(getNodeName(element)) >= 0;
}

function getDocumentElement(element) {
  // $FlowFixMe[incompatible-return]: assume body is always available
  return ((isElement(element) ? element.ownerDocument : // $FlowFixMe[prop-missing]
  element.document) || window.document).documentElement;
}

function getParentNode(element) {
  if (getNodeName(element) === 'html') {
    return element;
  }

  return (// this is a quicker (but less type safe) way to save quite some bytes from the bundle
    // $FlowFixMe[incompatible-return]
    // $FlowFixMe[prop-missing]
    element.assignedSlot || // step into the shadow DOM of the parent of a slotted node
    element.parentNode || ( // DOM Element detected
    isShadowRoot(element) ? element.host : null) || // ShadowRoot detected
    // $FlowFixMe[incompatible-call]: HTMLElement is a Node
    getDocumentElement(element) // fallback

  );
}

function getTrueOffsetParent(element) {
  if (!isHTMLElement(element) || // https://github.com/popperjs/popper-core/issues/837
  getComputedStyle$1(element).position === 'fixed') {
    return null;
  }

  return element.offsetParent;
} // `.offsetParent` reports `null` for fixed elements, while absolute elements
// return the containing block


function getContainingBlock(element) {
  var isFirefox = /firefox/i.test(getUAString());
  var isIE = /Trident/i.test(getUAString());

  if (isIE && isHTMLElement(element)) {
    // In IE 9, 10 and 11 fixed elements containing block is always established by the viewport
    var elementCss = getComputedStyle$1(element);

    if (elementCss.position === 'fixed') {
      return null;
    }
  }

  var currentNode = getParentNode(element);

  if (isShadowRoot(currentNode)) {
    currentNode = currentNode.host;
  }

  while (isHTMLElement(currentNode) && ['html', 'body'].indexOf(getNodeName(currentNode)) < 0) {
    var css = getComputedStyle$1(currentNode); // This is non-exhaustive but covers the most common CSS properties that
    // create a containing block.
    // https://developer.mozilla.org/en-US/docs/Web/CSS/Containing_block#identifying_the_containing_block

    if (css.transform !== 'none' || css.perspective !== 'none' || css.contain === 'paint' || ['transform', 'perspective'].indexOf(css.willChange) !== -1 || isFirefox && css.willChange === 'filter' || isFirefox && css.filter && css.filter !== 'none') {
      return currentNode;
    } else {
      currentNode = currentNode.parentNode;
    }
  }

  return null;
} // Gets the closest ancestor positioned element. Handles some edge cases,
// such as table ancestors and cross browser bugs.


function getOffsetParent(element) {
  var window = getWindow(element);
  var offsetParent = getTrueOffsetParent(element);

  while (offsetParent && isTableElement(offsetParent) && getComputedStyle$1(offsetParent).position === 'static') {
    offsetParent = getTrueOffsetParent(offsetParent);
  }

  if (offsetParent && (getNodeName(offsetParent) === 'html' || getNodeName(offsetParent) === 'body' && getComputedStyle$1(offsetParent).position === 'static')) {
    return window;
  }

  return offsetParent || getContainingBlock(element) || window;
}

function getMainAxisFromPlacement(placement) {
  return ['top', 'bottom'].indexOf(placement) >= 0 ? 'x' : 'y';
}

function within(min$1, value, max$1) {
  return max(min$1, min(value, max$1));
}
function withinMaxClamp(min, value, max) {
  var v = within(min, value, max);
  return v > max ? max : v;
}

function getFreshSideObject() {
  return {
    top: 0,
    right: 0,
    bottom: 0,
    left: 0
  };
}

function mergePaddingObject(paddingObject) {
  return Object.assign({}, getFreshSideObject(), paddingObject);
}

function expandToHashMap(value, keys) {
  return keys.reduce(function (hashMap, key) {
    hashMap[key] = value;
    return hashMap;
  }, {});
}

var toPaddingObject = function toPaddingObject(padding, state) {
  padding = typeof padding === 'function' ? padding(Object.assign({}, state.rects, {
    placement: state.placement
  })) : padding;
  return mergePaddingObject(typeof padding !== 'number' ? padding : expandToHashMap(padding, basePlacements));
};

function arrow(_ref) {
  var _state$modifiersData$;

  var state = _ref.state,
      name = _ref.name,
      options = _ref.options;
  var arrowElement = state.elements.arrow;
  var popperOffsets = state.modifiersData.popperOffsets;
  var basePlacement = getBasePlacement(state.placement);
  var axis = getMainAxisFromPlacement(basePlacement);
  var isVertical = [left, right].indexOf(basePlacement) >= 0;
  var len = isVertical ? 'height' : 'width';

  if (!arrowElement || !popperOffsets) {
    return;
  }

  var paddingObject = toPaddingObject(options.padding, state);
  var arrowRect = getLayoutRect(arrowElement);
  var minProp = axis === 'y' ? top : left;
  var maxProp = axis === 'y' ? bottom : right;
  var endDiff = state.rects.reference[len] + state.rects.reference[axis] - popperOffsets[axis] - state.rects.popper[len];
  var startDiff = popperOffsets[axis] - state.rects.reference[axis];
  var arrowOffsetParent = getOffsetParent(arrowElement);
  var clientSize = arrowOffsetParent ? axis === 'y' ? arrowOffsetParent.clientHeight || 0 : arrowOffsetParent.clientWidth || 0 : 0;
  var centerToReference = endDiff / 2 - startDiff / 2; // Make sure the arrow doesn't overflow the popper if the center point is
  // outside of the popper bounds

  var min = paddingObject[minProp];
  var max = clientSize - arrowRect[len] - paddingObject[maxProp];
  var center = clientSize / 2 - arrowRect[len] / 2 + centerToReference;
  var offset = within(min, center, max); // Prevents breaking syntax highlighting...

  var axisProp = axis;
  state.modifiersData[name] = (_state$modifiersData$ = {}, _state$modifiersData$[axisProp] = offset, _state$modifiersData$.centerOffset = offset - center, _state$modifiersData$);
}

function effect$1(_ref2) {
  var state = _ref2.state,
      options = _ref2.options;
  var _options$element = options.element,
      arrowElement = _options$element === void 0 ? '[data-popper-arrow]' : _options$element;

  if (arrowElement == null) {
    return;
  } // CSS selector


  if (typeof arrowElement === 'string') {
    arrowElement = state.elements.popper.querySelector(arrowElement);

    if (!arrowElement) {
      return;
    }
  }

  if (process.env.NODE_ENV !== "production") {
    if (!isHTMLElement(arrowElement)) {
      console.error(['Popper: "arrow" element must be an HTMLElement (not an SVGElement).', 'To use an SVG arrow, wrap it in an HTMLElement that will be used as', 'the arrow.'].join(' '));
    }
  }

  if (!contains(state.elements.popper, arrowElement)) {
    if (process.env.NODE_ENV !== "production") {
      console.error(['Popper: "arrow" modifier\'s `element` must be a child of the popper', 'element.'].join(' '));
    }

    return;
  }

  state.elements.arrow = arrowElement;
} // eslint-disable-next-line import/no-unused-modules


var arrow$1 = {
  name: 'arrow',
  enabled: true,
  phase: 'main',
  fn: arrow,
  effect: effect$1,
  requires: ['popperOffsets'],
  requiresIfExists: ['preventOverflow']
};

function getVariation(placement) {
  return placement.split('-')[1];
}

var unsetSides = {
  top: 'auto',
  right: 'auto',
  bottom: 'auto',
  left: 'auto'
}; // Round the offsets to the nearest suitable subpixel based on the DPR.
// Zooming can change the DPR, but it seems to report a value that will
// cleanly divide the values into the appropriate subpixels.

function roundOffsetsByDPR(_ref) {
  var x = _ref.x,
      y = _ref.y;
  var win = window;
  var dpr = win.devicePixelRatio || 1;
  return {
    x: round(x * dpr) / dpr || 0,
    y: round(y * dpr) / dpr || 0
  };
}

function mapToStyles(_ref2) {
  var _Object$assign2;

  var popper = _ref2.popper,
      popperRect = _ref2.popperRect,
      placement = _ref2.placement,
      variation = _ref2.variation,
      offsets = _ref2.offsets,
      position = _ref2.position,
      gpuAcceleration = _ref2.gpuAcceleration,
      adaptive = _ref2.adaptive,
      roundOffsets = _ref2.roundOffsets,
      isFixed = _ref2.isFixed;
  var _offsets$x = offsets.x,
      x = _offsets$x === void 0 ? 0 : _offsets$x,
      _offsets$y = offsets.y,
      y = _offsets$y === void 0 ? 0 : _offsets$y;

  var _ref3 = typeof roundOffsets === 'function' ? roundOffsets({
    x: x,
    y: y
  }) : {
    x: x,
    y: y
  };

  x = _ref3.x;
  y = _ref3.y;
  var hasX = offsets.hasOwnProperty('x');
  var hasY = offsets.hasOwnProperty('y');
  var sideX = left;
  var sideY = top;
  var win = window;

  if (adaptive) {
    var offsetParent = getOffsetParent(popper);
    var heightProp = 'clientHeight';
    var widthProp = 'clientWidth';

    if (offsetParent === getWindow(popper)) {
      offsetParent = getDocumentElement(popper);

      if (getComputedStyle$1(offsetParent).position !== 'static' && position === 'absolute') {
        heightProp = 'scrollHeight';
        widthProp = 'scrollWidth';
      }
    } // $FlowFixMe[incompatible-cast]: force type refinement, we compare offsetParent with window above, but Flow doesn't detect it


    offsetParent = offsetParent;

    if (placement === top || (placement === left || placement === right) && variation === end) {
      sideY = bottom;
      var offsetY = isFixed && offsetParent === win && win.visualViewport ? win.visualViewport.height : // $FlowFixMe[prop-missing]
      offsetParent[heightProp];
      y -= offsetY - popperRect.height;
      y *= gpuAcceleration ? 1 : -1;
    }

    if (placement === left || (placement === top || placement === bottom) && variation === end) {
      sideX = right;
      var offsetX = isFixed && offsetParent === win && win.visualViewport ? win.visualViewport.width : // $FlowFixMe[prop-missing]
      offsetParent[widthProp];
      x -= offsetX - popperRect.width;
      x *= gpuAcceleration ? 1 : -1;
    }
  }

  var commonStyles = Object.assign({
    position: position
  }, adaptive && unsetSides);

  var _ref4 = roundOffsets === true ? roundOffsetsByDPR({
    x: x,
    y: y
  }) : {
    x: x,
    y: y
  };

  x = _ref4.x;
  y = _ref4.y;

  if (gpuAcceleration) {
    var _Object$assign;

    return Object.assign({}, commonStyles, (_Object$assign = {}, _Object$assign[sideY] = hasY ? '0' : '', _Object$assign[sideX] = hasX ? '0' : '', _Object$assign.transform = (win.devicePixelRatio || 1) <= 1 ? "translate(" + x + "px, " + y + "px)" : "translate3d(" + x + "px, " + y + "px, 0)", _Object$assign));
  }

  return Object.assign({}, commonStyles, (_Object$assign2 = {}, _Object$assign2[sideY] = hasY ? y + "px" : '', _Object$assign2[sideX] = hasX ? x + "px" : '', _Object$assign2.transform = '', _Object$assign2));
}

function computeStyles(_ref5) {
  var state = _ref5.state,
      options = _ref5.options;
  var _options$gpuAccelerat = options.gpuAcceleration,
      gpuAcceleration = _options$gpuAccelerat === void 0 ? true : _options$gpuAccelerat,
      _options$adaptive = options.adaptive,
      adaptive = _options$adaptive === void 0 ? true : _options$adaptive,
      _options$roundOffsets = options.roundOffsets,
      roundOffsets = _options$roundOffsets === void 0 ? true : _options$roundOffsets;

  if (process.env.NODE_ENV !== "production") {
    var transitionProperty = getComputedStyle$1(state.elements.popper).transitionProperty || '';

    if (adaptive && ['transform', 'top', 'right', 'bottom', 'left'].some(function (property) {
      return transitionProperty.indexOf(property) >= 0;
    })) {
      console.warn(['Popper: Detected CSS transitions on at least one of the following', 'CSS properties: "transform", "top", "right", "bottom", "left".', '\n\n', 'Disable the "computeStyles" modifier\'s `adaptive` option to allow', 'for smooth transitions, or remove these properties from the CSS', 'transition declaration on the popper element if only transitioning', 'opacity or background-color for example.', '\n\n', 'We recommend using the popper element as a wrapper around an inner', 'element that can have any CSS property transitioned for animations.'].join(' '));
    }
  }

  var commonStyles = {
    placement: getBasePlacement(state.placement),
    variation: getVariation(state.placement),
    popper: state.elements.popper,
    popperRect: state.rects.popper,
    gpuAcceleration: gpuAcceleration,
    isFixed: state.options.strategy === 'fixed'
  };

  if (state.modifiersData.popperOffsets != null) {
    state.styles.popper = Object.assign({}, state.styles.popper, mapToStyles(Object.assign({}, commonStyles, {
      offsets: state.modifiersData.popperOffsets,
      position: state.options.strategy,
      adaptive: adaptive,
      roundOffsets: roundOffsets
    })));
  }

  if (state.modifiersData.arrow != null) {
    state.styles.arrow = Object.assign({}, state.styles.arrow, mapToStyles(Object.assign({}, commonStyles, {
      offsets: state.modifiersData.arrow,
      position: 'absolute',
      adaptive: false,
      roundOffsets: roundOffsets
    })));
  }

  state.attributes.popper = Object.assign({}, state.attributes.popper, {
    'data-popper-placement': state.placement
  });
} // eslint-disable-next-line import/no-unused-modules


var computeStyles$1 = {
  name: 'computeStyles',
  enabled: true,
  phase: 'beforeWrite',
  fn: computeStyles,
  data: {}
};

var passive = {
  passive: true
};

function effect(_ref) {
  var state = _ref.state,
      instance = _ref.instance,
      options = _ref.options;
  var _options$scroll = options.scroll,
      scroll = _options$scroll === void 0 ? true : _options$scroll,
      _options$resize = options.resize,
      resize = _options$resize === void 0 ? true : _options$resize;
  var window = getWindow(state.elements.popper);
  var scrollParents = [].concat(state.scrollParents.reference, state.scrollParents.popper);

  if (scroll) {
    scrollParents.forEach(function (scrollParent) {
      scrollParent.addEventListener('scroll', instance.update, passive);
    });
  }

  if (resize) {
    window.addEventListener('resize', instance.update, passive);
  }

  return function () {
    if (scroll) {
      scrollParents.forEach(function (scrollParent) {
        scrollParent.removeEventListener('scroll', instance.update, passive);
      });
    }

    if (resize) {
      window.removeEventListener('resize', instance.update, passive);
    }
  };
} // eslint-disable-next-line import/no-unused-modules


var eventListeners = {
  name: 'eventListeners',
  enabled: true,
  phase: 'write',
  fn: function fn() {},
  effect: effect,
  data: {}
};

var hash$1 = {
  left: 'right',
  right: 'left',
  bottom: 'top',
  top: 'bottom'
};
function getOppositePlacement(placement) {
  return placement.replace(/left|right|bottom|top/g, function (matched) {
    return hash$1[matched];
  });
}

var hash = {
  start: 'end',
  end: 'start'
};
function getOppositeVariationPlacement(placement) {
  return placement.replace(/start|end/g, function (matched) {
    return hash[matched];
  });
}

function getWindowScroll(node) {
  var win = getWindow(node);
  var scrollLeft = win.pageXOffset;
  var scrollTop = win.pageYOffset;
  return {
    scrollLeft: scrollLeft,
    scrollTop: scrollTop
  };
}

function getWindowScrollBarX(element) {
  // If <html> has a CSS width greater than the viewport, then this will be
  // incorrect for RTL.
  // Popper 1 is broken in this case and never had a bug report so let's assume
  // it's not an issue. I don't think anyone ever specifies width on <html>
  // anyway.
  // Browsers where the left scrollbar doesn't cause an issue report `0` for
  // this (e.g. Edge 2019, IE11, Safari)
  return getBoundingClientRect(getDocumentElement(element)).left + getWindowScroll(element).scrollLeft;
}

function getViewportRect(element, strategy) {
  var win = getWindow(element);
  var html = getDocumentElement(element);
  var visualViewport = win.visualViewport;
  var width = html.clientWidth;
  var height = html.clientHeight;
  var x = 0;
  var y = 0;

  if (visualViewport) {
    width = visualViewport.width;
    height = visualViewport.height;
    var layoutViewport = isLayoutViewport();

    if (layoutViewport || !layoutViewport && strategy === 'fixed') {
      x = visualViewport.offsetLeft;
      y = visualViewport.offsetTop;
    }
  }

  return {
    width: width,
    height: height,
    x: x + getWindowScrollBarX(element),
    y: y
  };
}

// of the `<html>` and `<body>` rect bounds if horizontally scrollable

function getDocumentRect(element) {
  var _element$ownerDocumen;

  var html = getDocumentElement(element);
  var winScroll = getWindowScroll(element);
  var body = (_element$ownerDocumen = element.ownerDocument) == null ? void 0 : _element$ownerDocumen.body;
  var width = max(html.scrollWidth, html.clientWidth, body ? body.scrollWidth : 0, body ? body.clientWidth : 0);
  var height = max(html.scrollHeight, html.clientHeight, body ? body.scrollHeight : 0, body ? body.clientHeight : 0);
  var x = -winScroll.scrollLeft + getWindowScrollBarX(element);
  var y = -winScroll.scrollTop;

  if (getComputedStyle$1(body || html).direction === 'rtl') {
    x += max(html.clientWidth, body ? body.clientWidth : 0) - width;
  }

  return {
    width: width,
    height: height,
    x: x,
    y: y
  };
}

function isScrollParent(element) {
  // Firefox wants us to check `-x` and `-y` variations as well
  var _getComputedStyle = getComputedStyle$1(element),
      overflow = _getComputedStyle.overflow,
      overflowX = _getComputedStyle.overflowX,
      overflowY = _getComputedStyle.overflowY;

  return /auto|scroll|overlay|hidden/.test(overflow + overflowY + overflowX);
}

function getScrollParent(node) {
  if (['html', 'body', '#document'].indexOf(getNodeName(node)) >= 0) {
    // $FlowFixMe[incompatible-return]: assume body is always available
    return node.ownerDocument.body;
  }

  if (isHTMLElement(node) && isScrollParent(node)) {
    return node;
  }

  return getScrollParent(getParentNode(node));
}

/*
given a DOM element, return the list of all scroll parents, up the list of ancesors
until we get to the top window object. This list is what we attach scroll listeners
to, because if any of these parent elements scroll, we'll need to re-calculate the
reference element's position.
*/

function listScrollParents(element, list) {
  var _element$ownerDocumen;

  if (list === void 0) {
    list = [];
  }

  var scrollParent = getScrollParent(element);
  var isBody = scrollParent === ((_element$ownerDocumen = element.ownerDocument) == null ? void 0 : _element$ownerDocumen.body);
  var win = getWindow(scrollParent);
  var target = isBody ? [win].concat(win.visualViewport || [], isScrollParent(scrollParent) ? scrollParent : []) : scrollParent;
  var updatedList = list.concat(target);
  return isBody ? updatedList : // $FlowFixMe[incompatible-call]: isBody tells us target will be an HTMLElement here
  updatedList.concat(listScrollParents(getParentNode(target)));
}

function rectToClientRect(rect) {
  return Object.assign({}, rect, {
    left: rect.x,
    top: rect.y,
    right: rect.x + rect.width,
    bottom: rect.y + rect.height
  });
}

function getInnerBoundingClientRect(element, strategy) {
  var rect = getBoundingClientRect(element, false, strategy === 'fixed');
  rect.top = rect.top + element.clientTop;
  rect.left = rect.left + element.clientLeft;
  rect.bottom = rect.top + element.clientHeight;
  rect.right = rect.left + element.clientWidth;
  rect.width = element.clientWidth;
  rect.height = element.clientHeight;
  rect.x = rect.left;
  rect.y = rect.top;
  return rect;
}

function getClientRectFromMixedType(element, clippingParent, strategy) {
  return clippingParent === viewport ? rectToClientRect(getViewportRect(element, strategy)) : isElement(clippingParent) ? getInnerBoundingClientRect(clippingParent, strategy) : rectToClientRect(getDocumentRect(getDocumentElement(element)));
} // A "clipping parent" is an overflowable container with the characteristic of
// clipping (or hiding) overflowing elements with a position different from
// `initial`


function getClippingParents(element) {
  var clippingParents = listScrollParents(getParentNode(element));
  var canEscapeClipping = ['absolute', 'fixed'].indexOf(getComputedStyle$1(element).position) >= 0;
  var clipperElement = canEscapeClipping && isHTMLElement(element) ? getOffsetParent(element) : element;

  if (!isElement(clipperElement)) {
    return [];
  } // $FlowFixMe[incompatible-return]: https://github.com/facebook/flow/issues/1414


  return clippingParents.filter(function (clippingParent) {
    return isElement(clippingParent) && contains(clippingParent, clipperElement) && getNodeName(clippingParent) !== 'body';
  });
} // Gets the maximum area that the element is visible in due to any number of
// clipping parents


function getClippingRect(element, boundary, rootBoundary, strategy) {
  var mainClippingParents = boundary === 'clippingParents' ? getClippingParents(element) : [].concat(boundary);
  var clippingParents = [].concat(mainClippingParents, [rootBoundary]);
  var firstClippingParent = clippingParents[0];
  var clippingRect = clippingParents.reduce(function (accRect, clippingParent) {
    var rect = getClientRectFromMixedType(element, clippingParent, strategy);
    accRect.top = max(rect.top, accRect.top);
    accRect.right = min(rect.right, accRect.right);
    accRect.bottom = min(rect.bottom, accRect.bottom);
    accRect.left = max(rect.left, accRect.left);
    return accRect;
  }, getClientRectFromMixedType(element, firstClippingParent, strategy));
  clippingRect.width = clippingRect.right - clippingRect.left;
  clippingRect.height = clippingRect.bottom - clippingRect.top;
  clippingRect.x = clippingRect.left;
  clippingRect.y = clippingRect.top;
  return clippingRect;
}

function computeOffsets(_ref) {
  var reference = _ref.reference,
      element = _ref.element,
      placement = _ref.placement;
  var basePlacement = placement ? getBasePlacement(placement) : null;
  var variation = placement ? getVariation(placement) : null;
  var commonX = reference.x + reference.width / 2 - element.width / 2;
  var commonY = reference.y + reference.height / 2 - element.height / 2;
  var offsets;

  switch (basePlacement) {
    case top:
      offsets = {
        x: commonX,
        y: reference.y - element.height
      };
      break;

    case bottom:
      offsets = {
        x: commonX,
        y: reference.y + reference.height
      };
      break;

    case right:
      offsets = {
        x: reference.x + reference.width,
        y: commonY
      };
      break;

    case left:
      offsets = {
        x: reference.x - element.width,
        y: commonY
      };
      break;

    default:
      offsets = {
        x: reference.x,
        y: reference.y
      };
  }

  var mainAxis = basePlacement ? getMainAxisFromPlacement(basePlacement) : null;

  if (mainAxis != null) {
    var len = mainAxis === 'y' ? 'height' : 'width';

    switch (variation) {
      case start:
        offsets[mainAxis] = offsets[mainAxis] - (reference[len] / 2 - element[len] / 2);
        break;

      case end:
        offsets[mainAxis] = offsets[mainAxis] + (reference[len] / 2 - element[len] / 2);
        break;
    }
  }

  return offsets;
}

function detectOverflow(state, options) {
  if (options === void 0) {
    options = {};
  }

  var _options = options,
      _options$placement = _options.placement,
      placement = _options$placement === void 0 ? state.placement : _options$placement,
      _options$strategy = _options.strategy,
      strategy = _options$strategy === void 0 ? state.strategy : _options$strategy,
      _options$boundary = _options.boundary,
      boundary = _options$boundary === void 0 ? clippingParents : _options$boundary,
      _options$rootBoundary = _options.rootBoundary,
      rootBoundary = _options$rootBoundary === void 0 ? viewport : _options$rootBoundary,
      _options$elementConte = _options.elementContext,
      elementContext = _options$elementConte === void 0 ? popper : _options$elementConte,
      _options$altBoundary = _options.altBoundary,
      altBoundary = _options$altBoundary === void 0 ? false : _options$altBoundary,
      _options$padding = _options.padding,
      padding = _options$padding === void 0 ? 0 : _options$padding;
  var paddingObject = mergePaddingObject(typeof padding !== 'number' ? padding : expandToHashMap(padding, basePlacements));
  var altContext = elementContext === popper ? reference : popper;
  var popperRect = state.rects.popper;
  var element = state.elements[altBoundary ? altContext : elementContext];
  var clippingClientRect = getClippingRect(isElement(element) ? element : element.contextElement || getDocumentElement(state.elements.popper), boundary, rootBoundary, strategy);
  var referenceClientRect = getBoundingClientRect(state.elements.reference);
  var popperOffsets = computeOffsets({
    reference: referenceClientRect,
    element: popperRect,
    strategy: 'absolute',
    placement: placement
  });
  var popperClientRect = rectToClientRect(Object.assign({}, popperRect, popperOffsets));
  var elementClientRect = elementContext === popper ? popperClientRect : referenceClientRect; // positive = overflowing the clipping rect
  // 0 or negative = within the clipping rect

  var overflowOffsets = {
    top: clippingClientRect.top - elementClientRect.top + paddingObject.top,
    bottom: elementClientRect.bottom - clippingClientRect.bottom + paddingObject.bottom,
    left: clippingClientRect.left - elementClientRect.left + paddingObject.left,
    right: elementClientRect.right - clippingClientRect.right + paddingObject.right
  };
  var offsetData = state.modifiersData.offset; // Offsets can be applied only to the popper element

  if (elementContext === popper && offsetData) {
    var offset = offsetData[placement];
    Object.keys(overflowOffsets).forEach(function (key) {
      var multiply = [right, bottom].indexOf(key) >= 0 ? 1 : -1;
      var axis = [top, bottom].indexOf(key) >= 0 ? 'y' : 'x';
      overflowOffsets[key] += offset[axis] * multiply;
    });
  }

  return overflowOffsets;
}

function computeAutoPlacement(state, options) {
  if (options === void 0) {
    options = {};
  }

  var _options = options,
      placement = _options.placement,
      boundary = _options.boundary,
      rootBoundary = _options.rootBoundary,
      padding = _options.padding,
      flipVariations = _options.flipVariations,
      _options$allowedAutoP = _options.allowedAutoPlacements,
      allowedAutoPlacements = _options$allowedAutoP === void 0 ? placements : _options$allowedAutoP;
  var variation = getVariation(placement);
  var placements$1 = variation ? flipVariations ? variationPlacements : variationPlacements.filter(function (placement) {
    return getVariation(placement) === variation;
  }) : basePlacements;
  var allowedPlacements = placements$1.filter(function (placement) {
    return allowedAutoPlacements.indexOf(placement) >= 0;
  });

  if (allowedPlacements.length === 0) {
    allowedPlacements = placements$1;

    if (process.env.NODE_ENV !== "production") {
      console.error(['Popper: The `allowedAutoPlacements` option did not allow any', 'placements. Ensure the `placement` option matches the variation', 'of the allowed placements.', 'For example, "auto" cannot be used to allow "bottom-start".', 'Use "auto-start" instead.'].join(' '));
    }
  } // $FlowFixMe[incompatible-type]: Flow seems to have problems with two array unions...


  var overflows = allowedPlacements.reduce(function (acc, placement) {
    acc[placement] = detectOverflow(state, {
      placement: placement,
      boundary: boundary,
      rootBoundary: rootBoundary,
      padding: padding
    })[getBasePlacement(placement)];
    return acc;
  }, {});
  return Object.keys(overflows).sort(function (a, b) {
    return overflows[a] - overflows[b];
  });
}

function getExpandedFallbackPlacements(placement) {
  if (getBasePlacement(placement) === auto) {
    return [];
  }

  var oppositePlacement = getOppositePlacement(placement);
  return [getOppositeVariationPlacement(placement), oppositePlacement, getOppositeVariationPlacement(oppositePlacement)];
}

function flip(_ref) {
  var state = _ref.state,
      options = _ref.options,
      name = _ref.name;

  if (state.modifiersData[name]._skip) {
    return;
  }

  var _options$mainAxis = options.mainAxis,
      checkMainAxis = _options$mainAxis === void 0 ? true : _options$mainAxis,
      _options$altAxis = options.altAxis,
      checkAltAxis = _options$altAxis === void 0 ? true : _options$altAxis,
      specifiedFallbackPlacements = options.fallbackPlacements,
      padding = options.padding,
      boundary = options.boundary,
      rootBoundary = options.rootBoundary,
      altBoundary = options.altBoundary,
      _options$flipVariatio = options.flipVariations,
      flipVariations = _options$flipVariatio === void 0 ? true : _options$flipVariatio,
      allowedAutoPlacements = options.allowedAutoPlacements;
  var preferredPlacement = state.options.placement;
  var basePlacement = getBasePlacement(preferredPlacement);
  var isBasePlacement = basePlacement === preferredPlacement;
  var fallbackPlacements = specifiedFallbackPlacements || (isBasePlacement || !flipVariations ? [getOppositePlacement(preferredPlacement)] : getExpandedFallbackPlacements(preferredPlacement));
  var placements = [preferredPlacement].concat(fallbackPlacements).reduce(function (acc, placement) {
    return acc.concat(getBasePlacement(placement) === auto ? computeAutoPlacement(state, {
      placement: placement,
      boundary: boundary,
      rootBoundary: rootBoundary,
      padding: padding,
      flipVariations: flipVariations,
      allowedAutoPlacements: allowedAutoPlacements
    }) : placement);
  }, []);
  var referenceRect = state.rects.reference;
  var popperRect = state.rects.popper;
  var checksMap = new Map();
  var makeFallbackChecks = true;
  var firstFittingPlacement = placements[0];

  for (var i = 0; i < placements.length; i++) {
    var placement = placements[i];

    var _basePlacement = getBasePlacement(placement);

    var isStartVariation = getVariation(placement) === start;
    var isVertical = [top, bottom].indexOf(_basePlacement) >= 0;
    var len = isVertical ? 'width' : 'height';
    var overflow = detectOverflow(state, {
      placement: placement,
      boundary: boundary,
      rootBoundary: rootBoundary,
      altBoundary: altBoundary,
      padding: padding
    });
    var mainVariationSide = isVertical ? isStartVariation ? right : left : isStartVariation ? bottom : top;

    if (referenceRect[len] > popperRect[len]) {
      mainVariationSide = getOppositePlacement(mainVariationSide);
    }

    var altVariationSide = getOppositePlacement(mainVariationSide);
    var checks = [];

    if (checkMainAxis) {
      checks.push(overflow[_basePlacement] <= 0);
    }

    if (checkAltAxis) {
      checks.push(overflow[mainVariationSide] <= 0, overflow[altVariationSide] <= 0);
    }

    if (checks.every(function (check) {
      return check;
    })) {
      firstFittingPlacement = placement;
      makeFallbackChecks = false;
      break;
    }

    checksMap.set(placement, checks);
  }

  if (makeFallbackChecks) {
    // `2` may be desired in some cases – research later
    var numberOfChecks = flipVariations ? 3 : 1;

    var _loop = function _loop(_i) {
      var fittingPlacement = placements.find(function (placement) {
        var checks = checksMap.get(placement);

        if (checks) {
          return checks.slice(0, _i).every(function (check) {
            return check;
          });
        }
      });

      if (fittingPlacement) {
        firstFittingPlacement = fittingPlacement;
        return "break";
      }
    };

    for (var _i = numberOfChecks; _i > 0; _i--) {
      var _ret = _loop(_i);

      if (_ret === "break") break;
    }
  }

  if (state.placement !== firstFittingPlacement) {
    state.modifiersData[name]._skip = true;
    state.placement = firstFittingPlacement;
    state.reset = true;
  }
} // eslint-disable-next-line import/no-unused-modules


var flip$1 = {
  name: 'flip',
  enabled: true,
  phase: 'main',
  fn: flip,
  requiresIfExists: ['offset'],
  data: {
    _skip: false
  }
};

function getSideOffsets(overflow, rect, preventedOffsets) {
  if (preventedOffsets === void 0) {
    preventedOffsets = {
      x: 0,
      y: 0
    };
  }

  return {
    top: overflow.top - rect.height - preventedOffsets.y,
    right: overflow.right - rect.width + preventedOffsets.x,
    bottom: overflow.bottom - rect.height + preventedOffsets.y,
    left: overflow.left - rect.width - preventedOffsets.x
  };
}

function isAnySideFullyClipped(overflow) {
  return [top, right, bottom, left].some(function (side) {
    return overflow[side] >= 0;
  });
}

function hide(_ref) {
  var state = _ref.state,
      name = _ref.name;
  var referenceRect = state.rects.reference;
  var popperRect = state.rects.popper;
  var preventedOffsets = state.modifiersData.preventOverflow;
  var referenceOverflow = detectOverflow(state, {
    elementContext: 'reference'
  });
  var popperAltOverflow = detectOverflow(state, {
    altBoundary: true
  });
  var referenceClippingOffsets = getSideOffsets(referenceOverflow, referenceRect);
  var popperEscapeOffsets = getSideOffsets(popperAltOverflow, popperRect, preventedOffsets);
  var isReferenceHidden = isAnySideFullyClipped(referenceClippingOffsets);
  var hasPopperEscaped = isAnySideFullyClipped(popperEscapeOffsets);
  state.modifiersData[name] = {
    referenceClippingOffsets: referenceClippingOffsets,
    popperEscapeOffsets: popperEscapeOffsets,
    isReferenceHidden: isReferenceHidden,
    hasPopperEscaped: hasPopperEscaped
  };
  state.attributes.popper = Object.assign({}, state.attributes.popper, {
    'data-popper-reference-hidden': isReferenceHidden,
    'data-popper-escaped': hasPopperEscaped
  });
} // eslint-disable-next-line import/no-unused-modules


var hide$1 = {
  name: 'hide',
  enabled: true,
  phase: 'main',
  requiresIfExists: ['preventOverflow'],
  fn: hide
};

function distanceAndSkiddingToXY(placement, rects, offset) {
  var basePlacement = getBasePlacement(placement);
  var invertDistance = [left, top].indexOf(basePlacement) >= 0 ? -1 : 1;

  var _ref = typeof offset === 'function' ? offset(Object.assign({}, rects, {
    placement: placement
  })) : offset,
      skidding = _ref[0],
      distance = _ref[1];

  skidding = skidding || 0;
  distance = (distance || 0) * invertDistance;
  return [left, right].indexOf(basePlacement) >= 0 ? {
    x: distance,
    y: skidding
  } : {
    x: skidding,
    y: distance
  };
}

function offset(_ref2) {
  var state = _ref2.state,
      options = _ref2.options,
      name = _ref2.name;
  var _options$offset = options.offset,
      offset = _options$offset === void 0 ? [0, 0] : _options$offset;
  var data = placements.reduce(function (acc, placement) {
    acc[placement] = distanceAndSkiddingToXY(placement, state.rects, offset);
    return acc;
  }, {});
  var _data$state$placement = data[state.placement],
      x = _data$state$placement.x,
      y = _data$state$placement.y;

  if (state.modifiersData.popperOffsets != null) {
    state.modifiersData.popperOffsets.x += x;
    state.modifiersData.popperOffsets.y += y;
  }

  state.modifiersData[name] = data;
} // eslint-disable-next-line import/no-unused-modules


var offset$1 = {
  name: 'offset',
  enabled: true,
  phase: 'main',
  requires: ['popperOffsets'],
  fn: offset
};

function popperOffsets(_ref) {
  var state = _ref.state,
      name = _ref.name;
  // Offsets are the actual position the popper needs to have to be
  // properly positioned near its reference element
  // This is the most basic placement, and will be adjusted by
  // the modifiers in the next step
  state.modifiersData[name] = computeOffsets({
    reference: state.rects.reference,
    element: state.rects.popper,
    strategy: 'absolute',
    placement: state.placement
  });
} // eslint-disable-next-line import/no-unused-modules


var popperOffsets$1 = {
  name: 'popperOffsets',
  enabled: true,
  phase: 'read',
  fn: popperOffsets,
  data: {}
};

function getAltAxis(axis) {
  return axis === 'x' ? 'y' : 'x';
}

function preventOverflow(_ref) {
  var state = _ref.state,
      options = _ref.options,
      name = _ref.name;
  var _options$mainAxis = options.mainAxis,
      checkMainAxis = _options$mainAxis === void 0 ? true : _options$mainAxis,
      _options$altAxis = options.altAxis,
      checkAltAxis = _options$altAxis === void 0 ? false : _options$altAxis,
      boundary = options.boundary,
      rootBoundary = options.rootBoundary,
      altBoundary = options.altBoundary,
      padding = options.padding,
      _options$tether = options.tether,
      tether = _options$tether === void 0 ? true : _options$tether,
      _options$tetherOffset = options.tetherOffset,
      tetherOffset = _options$tetherOffset === void 0 ? 0 : _options$tetherOffset;
  var overflow = detectOverflow(state, {
    boundary: boundary,
    rootBoundary: rootBoundary,
    padding: padding,
    altBoundary: altBoundary
  });
  var basePlacement = getBasePlacement(state.placement);
  var variation = getVariation(state.placement);
  var isBasePlacement = !variation;
  var mainAxis = getMainAxisFromPlacement(basePlacement);
  var altAxis = getAltAxis(mainAxis);
  var popperOffsets = state.modifiersData.popperOffsets;
  var referenceRect = state.rects.reference;
  var popperRect = state.rects.popper;
  var tetherOffsetValue = typeof tetherOffset === 'function' ? tetherOffset(Object.assign({}, state.rects, {
    placement: state.placement
  })) : tetherOffset;
  var normalizedTetherOffsetValue = typeof tetherOffsetValue === 'number' ? {
    mainAxis: tetherOffsetValue,
    altAxis: tetherOffsetValue
  } : Object.assign({
    mainAxis: 0,
    altAxis: 0
  }, tetherOffsetValue);
  var offsetModifierState = state.modifiersData.offset ? state.modifiersData.offset[state.placement] : null;
  var data = {
    x: 0,
    y: 0
  };

  if (!popperOffsets) {
    return;
  }

  if (checkMainAxis) {
    var _offsetModifierState$;

    var mainSide = mainAxis === 'y' ? top : left;
    var altSide = mainAxis === 'y' ? bottom : right;
    var len = mainAxis === 'y' ? 'height' : 'width';
    var offset = popperOffsets[mainAxis];
    var min$1 = offset + overflow[mainSide];
    var max$1 = offset - overflow[altSide];
    var additive = tether ? -popperRect[len] / 2 : 0;
    var minLen = variation === start ? referenceRect[len] : popperRect[len];
    var maxLen = variation === start ? -popperRect[len] : -referenceRect[len]; // We need to include the arrow in the calculation so the arrow doesn't go
    // outside the reference bounds

    var arrowElement = state.elements.arrow;
    var arrowRect = tether && arrowElement ? getLayoutRect(arrowElement) : {
      width: 0,
      height: 0
    };
    var arrowPaddingObject = state.modifiersData['arrow#persistent'] ? state.modifiersData['arrow#persistent'].padding : getFreshSideObject();
    var arrowPaddingMin = arrowPaddingObject[mainSide];
    var arrowPaddingMax = arrowPaddingObject[altSide]; // If the reference length is smaller than the arrow length, we don't want
    // to include its full size in the calculation. If the reference is small
    // and near the edge of a boundary, the popper can overflow even if the
    // reference is not overflowing as well (e.g. virtual elements with no
    // width or height)

    var arrowLen = within(0, referenceRect[len], arrowRect[len]);
    var minOffset = isBasePlacement ? referenceRect[len] / 2 - additive - arrowLen - arrowPaddingMin - normalizedTetherOffsetValue.mainAxis : minLen - arrowLen - arrowPaddingMin - normalizedTetherOffsetValue.mainAxis;
    var maxOffset = isBasePlacement ? -referenceRect[len] / 2 + additive + arrowLen + arrowPaddingMax + normalizedTetherOffsetValue.mainAxis : maxLen + arrowLen + arrowPaddingMax + normalizedTetherOffsetValue.mainAxis;
    var arrowOffsetParent = state.elements.arrow && getOffsetParent(state.elements.arrow);
    var clientOffset = arrowOffsetParent ? mainAxis === 'y' ? arrowOffsetParent.clientTop || 0 : arrowOffsetParent.clientLeft || 0 : 0;
    var offsetModifierValue = (_offsetModifierState$ = offsetModifierState == null ? void 0 : offsetModifierState[mainAxis]) != null ? _offsetModifierState$ : 0;
    var tetherMin = offset + minOffset - offsetModifierValue - clientOffset;
    var tetherMax = offset + maxOffset - offsetModifierValue;
    var preventedOffset = within(tether ? min(min$1, tetherMin) : min$1, offset, tether ? max(max$1, tetherMax) : max$1);
    popperOffsets[mainAxis] = preventedOffset;
    data[mainAxis] = preventedOffset - offset;
  }

  if (checkAltAxis) {
    var _offsetModifierState$2;

    var _mainSide = mainAxis === 'x' ? top : left;

    var _altSide = mainAxis === 'x' ? bottom : right;

    var _offset = popperOffsets[altAxis];

    var _len = altAxis === 'y' ? 'height' : 'width';

    var _min = _offset + overflow[_mainSide];

    var _max = _offset - overflow[_altSide];

    var isOriginSide = [top, left].indexOf(basePlacement) !== -1;

    var _offsetModifierValue = (_offsetModifierState$2 = offsetModifierState == null ? void 0 : offsetModifierState[altAxis]) != null ? _offsetModifierState$2 : 0;

    var _tetherMin = isOriginSide ? _min : _offset - referenceRect[_len] - popperRect[_len] - _offsetModifierValue + normalizedTetherOffsetValue.altAxis;

    var _tetherMax = isOriginSide ? _offset + referenceRect[_len] + popperRect[_len] - _offsetModifierValue - normalizedTetherOffsetValue.altAxis : _max;

    var _preventedOffset = tether && isOriginSide ? withinMaxClamp(_tetherMin, _offset, _tetherMax) : within(tether ? _tetherMin : _min, _offset, tether ? _tetherMax : _max);

    popperOffsets[altAxis] = _preventedOffset;
    data[altAxis] = _preventedOffset - _offset;
  }

  state.modifiersData[name] = data;
} // eslint-disable-next-line import/no-unused-modules


var preventOverflow$1 = {
  name: 'preventOverflow',
  enabled: true,
  phase: 'main',
  fn: preventOverflow,
  requiresIfExists: ['offset']
};

function getHTMLElementScroll(element) {
  return {
    scrollLeft: element.scrollLeft,
    scrollTop: element.scrollTop
  };
}

function getNodeScroll(node) {
  if (node === getWindow(node) || !isHTMLElement(node)) {
    return getWindowScroll(node);
  } else {
    return getHTMLElementScroll(node);
  }
}

function isElementScaled(element) {
  var rect = element.getBoundingClientRect();
  var scaleX = round(rect.width) / element.offsetWidth || 1;
  var scaleY = round(rect.height) / element.offsetHeight || 1;
  return scaleX !== 1 || scaleY !== 1;
} // Returns the composite rect of an element relative to its offsetParent.
// Composite means it takes into account transforms as well as layout.


function getCompositeRect(elementOrVirtualElement, offsetParent, isFixed) {
  if (isFixed === void 0) {
    isFixed = false;
  }

  var isOffsetParentAnElement = isHTMLElement(offsetParent);
  var offsetParentIsScaled = isHTMLElement(offsetParent) && isElementScaled(offsetParent);
  var documentElement = getDocumentElement(offsetParent);
  var rect = getBoundingClientRect(elementOrVirtualElement, offsetParentIsScaled, isFixed);
  var scroll = {
    scrollLeft: 0,
    scrollTop: 0
  };
  var offsets = {
    x: 0,
    y: 0
  };

  if (isOffsetParentAnElement || !isOffsetParentAnElement && !isFixed) {
    if (getNodeName(offsetParent) !== 'body' || // https://github.com/popperjs/popper-core/issues/1078
    isScrollParent(documentElement)) {
      scroll = getNodeScroll(offsetParent);
    }

    if (isHTMLElement(offsetParent)) {
      offsets = getBoundingClientRect(offsetParent, true);
      offsets.x += offsetParent.clientLeft;
      offsets.y += offsetParent.clientTop;
    } else if (documentElement) {
      offsets.x = getWindowScrollBarX(documentElement);
    }
  }

  return {
    x: rect.left + scroll.scrollLeft - offsets.x,
    y: rect.top + scroll.scrollTop - offsets.y,
    width: rect.width,
    height: rect.height
  };
}

function order(modifiers) {
  var map = new Map();
  var visited = new Set();
  var result = [];
  modifiers.forEach(function (modifier) {
    map.set(modifier.name, modifier);
  }); // On visiting object, check for its dependencies and visit them recursively

  function sort(modifier) {
    visited.add(modifier.name);
    var requires = [].concat(modifier.requires || [], modifier.requiresIfExists || []);
    requires.forEach(function (dep) {
      if (!visited.has(dep)) {
        var depModifier = map.get(dep);

        if (depModifier) {
          sort(depModifier);
        }
      }
    });
    result.push(modifier);
  }

  modifiers.forEach(function (modifier) {
    if (!visited.has(modifier.name)) {
      // check for visited object
      sort(modifier);
    }
  });
  return result;
}

function orderModifiers(modifiers) {
  // order based on dependencies
  var orderedModifiers = order(modifiers); // order based on phase

  return modifierPhases.reduce(function (acc, phase) {
    return acc.concat(orderedModifiers.filter(function (modifier) {
      return modifier.phase === phase;
    }));
  }, []);
}

function debounce(fn) {
  var pending;
  return function () {
    if (!pending) {
      pending = new Promise(function (resolve) {
        Promise.resolve().then(function () {
          pending = undefined;
          resolve(fn());
        });
      });
    }

    return pending;
  };
}

function format(str) {
  for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
    args[_key - 1] = arguments[_key];
  }

  return [].concat(args).reduce(function (p, c) {
    return p.replace(/%s/, c);
  }, str);
}

var INVALID_MODIFIER_ERROR = 'Popper: modifier "%s" provided an invalid %s property, expected %s but got %s';
var MISSING_DEPENDENCY_ERROR = 'Popper: modifier "%s" requires "%s", but "%s" modifier is not available';
var VALID_PROPERTIES = ['name', 'enabled', 'phase', 'fn', 'effect', 'requires', 'options'];
function validateModifiers(modifiers) {
  modifiers.forEach(function (modifier) {
    [].concat(Object.keys(modifier), VALID_PROPERTIES) // IE11-compatible replacement for `new Set(iterable)`
    .filter(function (value, index, self) {
      return self.indexOf(value) === index;
    }).forEach(function (key) {
      switch (key) {
        case 'name':
          if (typeof modifier.name !== 'string') {
            console.error(format(INVALID_MODIFIER_ERROR, String(modifier.name), '"name"', '"string"', "\"" + String(modifier.name) + "\""));
          }

          break;

        case 'enabled':
          if (typeof modifier.enabled !== 'boolean') {
            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"enabled"', '"boolean"', "\"" + String(modifier.enabled) + "\""));
          }

          break;

        case 'phase':
          if (modifierPhases.indexOf(modifier.phase) < 0) {
            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"phase"', "either " + modifierPhases.join(', '), "\"" + String(modifier.phase) + "\""));
          }

          break;

        case 'fn':
          if (typeof modifier.fn !== 'function') {
            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"fn"', '"function"', "\"" + String(modifier.fn) + "\""));
          }

          break;

        case 'effect':
          if (modifier.effect != null && typeof modifier.effect !== 'function') {
            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"effect"', '"function"', "\"" + String(modifier.fn) + "\""));
          }

          break;

        case 'requires':
          if (modifier.requires != null && !Array.isArray(modifier.requires)) {
            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"requires"', '"array"', "\"" + String(modifier.requires) + "\""));
          }

          break;

        case 'requiresIfExists':
          if (!Array.isArray(modifier.requiresIfExists)) {
            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"requiresIfExists"', '"array"', "\"" + String(modifier.requiresIfExists) + "\""));
          }

          break;

        case 'options':
        case 'data':
          break;

        default:
          console.error("PopperJS: an invalid property has been provided to the \"" + modifier.name + "\" modifier, valid properties are " + VALID_PROPERTIES.map(function (s) {
            return "\"" + s + "\"";
          }).join(', ') + "; but \"" + key + "\" was provided.");
      }

      modifier.requires && modifier.requires.forEach(function (requirement) {
        if (modifiers.find(function (mod) {
          return mod.name === requirement;
        }) == null) {
          console.error(format(MISSING_DEPENDENCY_ERROR, String(modifier.name), requirement, requirement));
        }
      });
    });
  });
}

function uniqueBy(arr, fn) {
  var identifiers = new Set();
  return arr.filter(function (item) {
    var identifier = fn(item);

    if (!identifiers.has(identifier)) {
      identifiers.add(identifier);
      return true;
    }
  });
}

function mergeByName(modifiers) {
  var merged = modifiers.reduce(function (merged, current) {
    var existing = merged[current.name];
    merged[current.name] = existing ? Object.assign({}, existing, current, {
      options: Object.assign({}, existing.options, current.options),
      data: Object.assign({}, existing.data, current.data)
    }) : current;
    return merged;
  }, {}); // IE11 does not support Object.values

  return Object.keys(merged).map(function (key) {
    return merged[key];
  });
}

var INVALID_ELEMENT_ERROR = 'Popper: Invalid reference or popper argument provided. They must be either a DOM element or virtual element.';
var INFINITE_LOOP_ERROR = 'Popper: An infinite loop in the modifiers cycle has been detected! The cycle has been interrupted to prevent a browser crash.';
var DEFAULT_OPTIONS = {
  placement: 'bottom',
  modifiers: [],
  strategy: 'absolute'
};

function areValidElements() {
  for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
    args[_key] = arguments[_key];
  }

  return !args.some(function (element) {
    return !(element && typeof element.getBoundingClientRect === 'function');
  });
}

function popperGenerator(generatorOptions) {
  if (generatorOptions === void 0) {
    generatorOptions = {};
  }

  var _generatorOptions = generatorOptions,
      _generatorOptions$def = _generatorOptions.defaultModifiers,
      defaultModifiers = _generatorOptions$def === void 0 ? [] : _generatorOptions$def,
      _generatorOptions$def2 = _generatorOptions.defaultOptions,
      defaultOptions = _generatorOptions$def2 === void 0 ? DEFAULT_OPTIONS : _generatorOptions$def2;
  return function createPopper(reference, popper, options) {
    if (options === void 0) {
      options = defaultOptions;
    }

    var state = {
      placement: 'bottom',
      orderedModifiers: [],
      options: Object.assign({}, DEFAULT_OPTIONS, defaultOptions),
      modifiersData: {},
      elements: {
        reference: reference,
        popper: popper
      },
      attributes: {},
      styles: {}
    };
    var effectCleanupFns = [];
    var isDestroyed = false;
    var instance = {
      state: state,
      setOptions: function setOptions(setOptionsAction) {
        var options = typeof setOptionsAction === 'function' ? setOptionsAction(state.options) : setOptionsAction;
        cleanupModifierEffects();
        state.options = Object.assign({}, defaultOptions, state.options, options);
        state.scrollParents = {
          reference: isElement(reference) ? listScrollParents(reference) : reference.contextElement ? listScrollParents(reference.contextElement) : [],
          popper: listScrollParents(popper)
        }; // Orders the modifiers based on their dependencies and `phase`
        // properties

        var orderedModifiers = orderModifiers(mergeByName([].concat(defaultModifiers, state.options.modifiers))); // Strip out disabled modifiers

        state.orderedModifiers = orderedModifiers.filter(function (m) {
          return m.enabled;
        }); // Validate the provided modifiers so that the consumer will get warned
        // if one of the modifiers is invalid for any reason

        if (process.env.NODE_ENV !== "production") {
          var modifiers = uniqueBy([].concat(orderedModifiers, state.options.modifiers), function (_ref) {
            var name = _ref.name;
            return name;
          });
          validateModifiers(modifiers);

          if (getBasePlacement(state.options.placement) === auto) {
            var flipModifier = state.orderedModifiers.find(function (_ref2) {
              var name = _ref2.name;
              return name === 'flip';
            });

            if (!flipModifier) {
              console.error(['Popper: "auto" placements require the "flip" modifier be', 'present and enabled to work.'].join(' '));
            }
          }

          var _getComputedStyle = getComputedStyle$1(popper),
              marginTop = _getComputedStyle.marginTop,
              marginRight = _getComputedStyle.marginRight,
              marginBottom = _getComputedStyle.marginBottom,
              marginLeft = _getComputedStyle.marginLeft; // We no longer take into account `margins` on the popper, and it can
          // cause bugs with positioning, so we'll warn the consumer


          if ([marginTop, marginRight, marginBottom, marginLeft].some(function (margin) {
            return parseFloat(margin);
          })) {
            console.warn(['Popper: CSS "margin" styles cannot be used to apply padding', 'between the popper and its reference element or boundary.', 'To replicate margin, use the `offset` modifier, as well as', 'the `padding` option in the `preventOverflow` and `flip`', 'modifiers.'].join(' '));
          }
        }

        runModifierEffects();
        return instance.update();
      },
      // Sync update – it will always be executed, even if not necessary. This
      // is useful for low frequency updates where sync behavior simplifies the
      // logic.
      // For high frequency updates (e.g. `resize` and `scroll` events), always
      // prefer the async Popper#update method
      forceUpdate: function forceUpdate() {
        if (isDestroyed) {
          return;
        }

        var _state$elements = state.elements,
            reference = _state$elements.reference,
            popper = _state$elements.popper; // Don't proceed if `reference` or `popper` are not valid elements
        // anymore

        if (!areValidElements(reference, popper)) {
          if (process.env.NODE_ENV !== "production") {
            console.error(INVALID_ELEMENT_ERROR);
          }

          return;
        } // Store the reference and popper rects to be read by modifiers


        state.rects = {
          reference: getCompositeRect(reference, getOffsetParent(popper), state.options.strategy === 'fixed'),
          popper: getLayoutRect(popper)
        }; // Modifiers have the ability to reset the current update cycle. The
        // most common use case for this is the `flip` modifier changing the
        // placement, which then needs to re-run all the modifiers, because the
        // logic was previously ran for the previous placement and is therefore
        // stale/incorrect

        state.reset = false;
        state.placement = state.options.placement; // On each update cycle, the `modifiersData` property for each modifier
        // is filled with the initial data specified by the modifier. This means
        // it doesn't persist and is fresh on each update.
        // To ensure persistent data, use `${name}#persistent`

        state.orderedModifiers.forEach(function (modifier) {
          return state.modifiersData[modifier.name] = Object.assign({}, modifier.data);
        });
        var __debug_loops__ = 0;

        for (var index = 0; index < state.orderedModifiers.length; index++) {
          if (process.env.NODE_ENV !== "production") {
            __debug_loops__ += 1;

            if (__debug_loops__ > 100) {
              console.error(INFINITE_LOOP_ERROR);
              break;
            }
          }

          if (state.reset === true) {
            state.reset = false;
            index = -1;
            continue;
          }

          var _state$orderedModifie = state.orderedModifiers[index],
              fn = _state$orderedModifie.fn,
              _state$orderedModifie2 = _state$orderedModifie.options,
              _options = _state$orderedModifie2 === void 0 ? {} : _state$orderedModifie2,
              name = _state$orderedModifie.name;

          if (typeof fn === 'function') {
            state = fn({
              state: state,
              options: _options,
              name: name,
              instance: instance
            }) || state;
          }
        }
      },
      // Async and optimistically optimized update – it will not be executed if
      // not necessary (debounced to run at most once-per-tick)
      update: debounce(function () {
        return new Promise(function (resolve) {
          instance.forceUpdate();
          resolve(state);
        });
      }),
      destroy: function destroy() {
        cleanupModifierEffects();
        isDestroyed = true;
      }
    };

    if (!areValidElements(reference, popper)) {
      if (process.env.NODE_ENV !== "production") {
        console.error(INVALID_ELEMENT_ERROR);
      }

      return instance;
    }

    instance.setOptions(options).then(function (state) {
      if (!isDestroyed && options.onFirstUpdate) {
        options.onFirstUpdate(state);
      }
    }); // Modifiers have the ability to execute arbitrary code before the first
    // update cycle runs. They will be executed in the same order as the update
    // cycle. This is useful when a modifier adds some persistent data that
    // other modifiers need to use, but the modifier is run after the dependent
    // one.

    function runModifierEffects() {
      state.orderedModifiers.forEach(function (_ref3) {
        var name = _ref3.name,
            _ref3$options = _ref3.options,
            options = _ref3$options === void 0 ? {} : _ref3$options,
            effect = _ref3.effect;

        if (typeof effect === 'function') {
          var cleanupFn = effect({
            state: state,
            name: name,
            instance: instance,
            options: options
          });

          var noopFn = function noopFn() {};

          effectCleanupFns.push(cleanupFn || noopFn);
        }
      });
    }

    function cleanupModifierEffects() {
      effectCleanupFns.forEach(function (fn) {
        return fn();
      });
      effectCleanupFns = [];
    }

    return instance;
  };
}

var defaultModifiers = [eventListeners, popperOffsets$1, computeStyles$1, applyStyles$1, offset$1, flip$1, preventOverflow$1, arrow$1, hide$1];
var createPopper = /*#__PURE__*/popperGenerator({
  defaultModifiers: defaultModifiers
}); // eslint-disable-next-line import/no-unused-modules

const CDropdown = defineComponent({
    name: 'CDropdown',
    props: {
        /**
         * Set aligment of dropdown menu.
         *
         * @values { 'start' | 'end' | { xs: 'start' | 'end' } | { sm: 'start' | 'end' } | { md: 'start' | 'end' } | { lg: 'start' | 'end' } | { xl: 'start' | 'end'} | { xxl: 'start' | 'end'} }
         */
        alignment: {
            type: [String, Object],
            default: undefined,
            required: false,
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            validator: (value) => {
                if (value === 'start' || value === 'end') {
                    return true;
                }
                else {
                    if (typeof value.xs !== 'undefined' && (value.xs === 'start' || value.xs === 'end')) {
                        return true;
                    }
                    if (typeof value.sm !== 'undefined' && (value.sm === 'start' || value.sm === 'end')) {
                        return true;
                    }
                    if (typeof value.md !== 'undefined' && (value.md === 'start' || value.md === 'end')) {
                        return true;
                    }
                    if (typeof value.lg !== 'undefined' && (value.lg === 'start' || value.lg === 'end')) {
                        return true;
                    }
                    if (typeof value.xl !== 'undefined' && (value.xl === 'start' || value.xl === 'end')) {
                        return true;
                    }
                    if (typeof value.xxl !== 'undefined' && (value.xxl === 'start' || value.xxl === 'end')) {
                        return true;
                    }
                    return false;
                }
            },
        },
        /**
         * Configure the auto close behavior of the dropdown:
         * - `true` - the dropdown will be closed by clicking outside or inside the dropdown menu.
         * - `false` - the dropdown will be closed by clicking the toggle button and manually calling hide or toggle method. (Also will not be closed by pressing esc key)
         * - `'inside'` - the dropdown will be closed (only) by clicking inside the dropdown menu.
         * - `'outside'` - the dropdown will be closed (only) by clicking outside the dropdown menu.
         */
        autoClose: {
            type: [Boolean, String],
            default: true,
            validator: (value) => {
                return typeof value === 'boolean' || ['inside', 'outside'].includes(value);
            },
        },
        /**
         * Sets a darker color scheme to match a dark navbar.
         */
        dark: {
            type: Boolean,
            required: false,
        },
        /**
         * Sets a specified  direction and location of the dropdown menu.
         *
         * @values 'center', 'dropup', 'dropup-center', 'dropend', 'dropstart'
         */
        direction: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['center', 'dropup', 'dropup-center', 'dropend', 'dropstart'].includes(value);
            },
        },
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * Describes the placement of your component after Popper.js has applied all the modifiers that may have flipped or altered the originally provided placement property.
         *
         * @values 'auto', 'top-end', 'top', 'top-start', 'bottom-end', 'bottom', 'bottom-start', 'right-start', 'right', 'right-end', 'left-start', 'left', 'left-end'
         */
        placement: {
            type: String,
            default: 'bottom-start',
            required: false,
        },
        /**
         * If you want to disable dynamic positioning set this property to `true`.
         */
        popper: {
            type: Boolean,
            default: true,
            required: false,
        },
        /**
         * Sets which event handlers you’d like provided to your toggle prop. You can specify one trigger or an array of them.
         */
        trigger: {
            type: [String, Array],
            required: false,
            default: 'click',
        },
        /**
         * Set the dropdown variant to an btn-group, dropdown, input-group, and nav-item.
         *
         * @values 'btn-group', 'dropdown', 'input-group', 'nav-item'
         */
        variant: {
            type: String,
            default: 'btn-group',
            required: false,
            validator: (value) => {
                return ['btn-group', 'dropdown', 'input-group', 'nav-item'].includes(value);
            },
        },
        /**
         * Toggle the visibility of dropdown menu component.
         */
        visible: {
            type: Boolean,
            required: false,
        },
    },
    emits: [
        /**
         * Callback fired when the component requests to be hidden.
         */
        'hide',
        /**
         * Callback fired when the component requests to be shown.
         */
        'show',
    ],
    setup(props, { slots, emit }) {
        const dropdownToggleRef = ref();
        const dropdownMenuRef = ref();
        const placement = ref(props.placement);
        const popper = ref();
        const visible = ref(props.visible);
        watch(() => props.visible, () => {
            visible.value = props.visible;
        });
        provide('config', {
            autoClose: props.autoClose,
            alignment: props.alignment,
            dark: props.dark,
            popper: props.popper,
        });
        provide('variant', props.variant);
        provide('visible', visible);
        provide('dropdownToggleRef', dropdownToggleRef);
        provide('dropdownMenuRef', dropdownMenuRef);
        if (props.direction === 'center') {
            placement.value = 'bottom';
        }
        if (props.direction === 'dropup') {
            placement.value = 'top-start';
        }
        if (props.direction === 'dropup-center') {
            placement.value = 'top';
        }
        if (props.direction === 'dropend') {
            placement.value = 'right-start';
        }
        if (props.direction === 'dropstart') {
            placement.value = 'left-start';
        }
        if (props.alignment === 'end') {
            placement.value = 'bottom-end';
        }
        const initPopper = () => {
            // Disable popper if responsive aligment is set.
            if (typeof props.alignment === 'object') {
                return;
            }
            if (dropdownToggleRef.value) {
                popper.value = createPopper(dropdownToggleRef.value, dropdownMenuRef.value, {
                    placement: placement.value,
                });
            }
        };
        const destroyPopper = () => {
            if (popper.value) {
                popper.value.destroy();
            }
            popper.value = undefined;
        };
        const toggleMenu = (_visible) => {
            if (props.disabled) {
                return;
            }
            if (typeof _visible == 'boolean') {
                visible.value = _visible;
                return;
            }
            if (visible.value === true) {
                visible.value = false;
                return;
            }
            visible.value = true;
        };
        provide('toggleMenu', toggleMenu);
        const hideMenu = () => {
            if (props.disabled) {
                return;
            }
            visible.value = false;
        };
        provide('hideMenu', hideMenu);
        watch(visible, () => {
            props.popper && (visible.value ? initPopper() : destroyPopper());
            visible.value ? emit('show') : emit('hide');
        });
        return () => props.variant === 'input-group'
            ? [slots.default && slots.default()]
            : h$1('div', {
                class: [
                    props.variant === 'nav-item' ? 'nav-item dropdown' : props.variant,
                    props.direction === 'center'
                        ? 'dropdown-center'
                        : props.direction === 'dropup-center'
                            ? 'dropup dropup-center'
                            : props.direction,
                ],
            }, slots.default && slots.default());
    },
});

const CDropdownItem = defineComponent({
    name: 'CDropdownItem',
    props: {
        /**
         * Toggle the active state for the component.
         */
        active: {
            type: Boolean,
            required: false,
        },
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'a',
            required: false,
        },
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * The href attribute specifies the URL of the page the link goes to.
         */
        href: {
            type: String,
            default: undefined,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(CLink, {
            class: 'dropdown-item',
            active: props.active,
            component: props.component,
            disabled: props.disabled,
            href: props.href,
        }, {
            default: () => slots.default && slots.default(),
        });
    },
});

const CDropdownHeader = defineComponent({
    name: 'CDropdownHeader',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'h6',
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: 'dropdown-header',
        }, slots.default && slots.default());
    },
});

const CDropdownDivider = defineComponent({
    name: 'CDropdownDivider',
    setup() {
        return () => h$1('hr', {
            class: 'dropdown-divider',
        });
    },
});

const CDropdownMenu = defineComponent({
    name: 'CDropdownMenu',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         *
         * @values 'div', 'ul'
         */
        component: {
            type: String,
            default: 'div',
            required: false,
        },
    },
    setup(props, { slots }) {
        const dropdownToggleRef = inject('dropdownToggleRef');
        const dropdownMenuRef = inject('dropdownMenuRef');
        const config = inject('config'); // eslint-disable-line @typescript-eslint/no-explicit-any
        const hideMenu = inject('hideMenu');
        const visible = inject('visible');
        const { autoClose, alignment, dark, popper } = config;
        // eslint-disable-next-line @typescript-eslint/ban-types
        const alignmentClassNames = (alignment) => {
            const classNames = [];
            if (typeof alignment === 'object') {
                Object.keys(alignment).map((key) => {
                    classNames.push(`dropdown-menu${key === 'xs' ? '' : `-${key}`}-${alignment[key]}`);
                });
            }
            if (typeof alignment === 'string') {
                classNames.push(`dropdown-menu-${alignment}`);
            }
            return classNames;
        };
        const handleKeyup = (event) => {
            if (autoClose === false) {
                return;
            }
            if (event.key === 'Escape') {
                hideMenu();
            }
        };
        const handleMouseUp = (event) => {
            if (dropdownToggleRef.value?.contains(event.target)) {
                return;
            }
            if (autoClose === true) {
                hideMenu();
                return;
            }
            if (autoClose === 'inside' && dropdownMenuRef.value?.contains(event.target)) {
                hideMenu();
                return;
            }
            if (autoClose === 'outside' &&
                !dropdownMenuRef.value?.contains(event.target)) {
                hideMenu();
            }
        };
        onUpdated(() => {
            visible.value && window.addEventListener('mouseup', handleMouseUp);
            visible.value && window.addEventListener('keyup', handleKeyup);
        });
        onUnmounted(() => {
            window.removeEventListener('mouseup', handleMouseUp);
            window.removeEventListener('keyup', handleKeyup);
        });
        return () => h$1(props.component, {
            class: [
                'dropdown-menu',
                { 'dropdown-menu-dark': dark, show: visible.value },
                alignmentClassNames(alignment),
            ],
            ...((typeof alignment === 'object' || !popper) && {
                'data-coreui-popper': 'static',
            }),
            ref: dropdownMenuRef,
        }, props.component === 'ul'
            ? slots.default && slots.default().map((vnode) => h$1('li', {}, vnode))
            : slots.default && slots.default());
    },
});

const CDropdownToggle = defineComponent({
    name: 'CDropdownToggle',
    props: {
        /**
         * Toggle the active state for the component.
         */
        active: {
            type: Boolean,
            default: false,
            required: false,
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Enables pseudo element caret on toggler.
         */
        caret: {
            type: Boolean,
            default: true,
            required: false,
        },
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'button',
            require: false,
        },
        /**
         * Create a custom toggler which accepts any content.
         */
        custom: Boolean,
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * @values 'rounded', 'rounded-top', 'rounded-end', 'rounded-bottom', 'rounded-start', 'rounded-circle', 'rounded-pill', 'rounded-0', 'rounded-1', 'rounded-2', 'rounded-3'
         */
        shape: Shape,
        /**
         * Size the component small or large.
         *
         * @values 'sm', 'lg'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['sm', 'lg'].includes(value);
            },
        },
        /**
         * Similarly, create split button dropdowns with virtually the same markup as single button dropdowns, but with the addition of `.dropdown-toggle-split` className for proper spacing around the dropdown caret.
         */
        split: {
            type: Boolean,
            required: false,
        },
        /**
         * Sets which event handlers you’d like provided to your toggle prop. You can specify one trigger or an array of them.
         *
         * @type 'hover' | 'focus' | 'click'
         */
        trigger: {
            type: String,
            default: 'click',
            required: false,
        },
        /**
         * Set the button variant to an outlined button or a ghost button.
         *
         * @values 'ghost', 'outline'
         */
        variant: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['ghost', 'outline'].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        const togglerRef = ref();
        const dropdownToggleRef = inject('dropdownToggleRef');
        const dropdownVariant = inject('variant');
        const visible = inject('visible');
        const toggleMenu = inject('toggleMenu');
        const className = [
            {
                'dropdown-toggle': props.caret,
                'dropdown-toggle-split': props.split,
                active: props.active,
                disabled: props.disabled,
            },
        ];
        const triggers = {
            ...((props.trigger === 'click' || props.trigger.includes('click')) && {
                onClick: () => {
                    if (props.disabled) {
                        return;
                    }
                    toggleMenu();
                },
            }),
            ...((props.trigger === 'focus' || props.trigger.includes('focus')) && {
                onfocus: () => {
                    if (props.disabled) {
                        return;
                    }
                    toggleMenu(true);
                },
                onblur: () => {
                    if (props.disabled) {
                        return;
                    }
                    toggleMenu(false);
                },
            }),
        };
        onMounted(() => {
            if (togglerRef.value) {
                dropdownToggleRef.value = togglerRef.value.$el;
            }
        });
        return () => props.custom
            ? slots.default &&
                slots.default().map((slot) => cloneVNode(slot, {
                    ref: (el) => {
                        togglerRef.value = el;
                    },
                    ...triggers,
                }))
            : dropdownVariant === 'nav-item'
                ? h$1('a', {
                    active: props.active,
                    class: [
                        'nav-link',
                        className,
                        {
                            show: visible.value,
                        },
                    ],
                    disabled: props.disabled,
                    href: '#',
                    ref: dropdownToggleRef,
                    ...triggers,
                }, { default: () => slots.default && slots.default() })
                : h$1(CButton, {
                    class: [
                        className,
                        {
                            show: visible.value,
                        },
                    ],
                    active: props.active,
                    color: props.color,
                    disabled: props.disabled,
                    ref: (el) => {
                        togglerRef.value = el;
                    },
                    shape: props.shape,
                    size: props.size,
                    ...triggers,
                    ...(props.component === 'button' && { type: 'button' }),
                    variant: props.variant,
                }, () => props.split
                    ? h$1('span', { class: 'visually-hidden' }, 'Toggle Dropdown')
                    : slots.default && slots.default());
    },
});

const CDropdownPlugin = {
    install: (app) => {
        app.component(CDropdown.name, CDropdown);
        app.component(CDropdownItem.name, CDropdownItem);
        app.component(CDropdownHeader.name, CDropdownHeader);
        app.component(CDropdownDivider.name, CDropdownDivider);
        app.component(CDropdownMenu.name, CDropdownMenu);
        app.component(CDropdownToggle.name, CDropdownToggle);
    },
};

const CFooter = defineComponent({
    name: 'CFooter',
    props: {
        /**
         * Place footer in non-static positions.
         *
         * @values 'fixed', 'sticky'
         */
        position: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['fixed', 'sticky'].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1('div', { class: ['footer', { [`footer-${props.position}`]: props.position }] }, slots.default && slots.default());
    },
});

const CFooterPlugin = {
    install: (app) => {
        app.component(CFooter.name, CFooter);
    },
};

const CForm = defineComponent({
    name: 'CForm',
    props: {
        /**
         * Mark a form as validated. If you set it `true`, all validation styles will be applied to the forms component.
         */
        validated: {
            type: Boolean,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1('form', { class: [{ ['was-validated']: props.validated }] }, slots.default && slots.default());
    },
});

const CFormFeedback = defineComponent({
    name: 'CFormFeedback',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'div',
        },
        /**
         * Method called immediately after the `value` prop changes.
         */
        invalid: Boolean,
        /**
         * If your form layout allows it, you can display validation feedback in a styled tooltip.
         */
        tooltip: Boolean,
        /**
         * Set component validation state to valid.
         */
        valid: Boolean,
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: [
                {
                    [`invalid-${props.tooltip ? 'tooltip' : 'feedback'}`]: props.invalid,
                    [`valid-${props.tooltip ? 'tooltip' : 'feedback'}`]: props.valid,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CFormControlValidation = defineComponent({
    name: 'CFormControlValidation',
    inheritAttrs: false,
    props: {
        /**
         * @ignore
         */
        describedby: {
            type: String,
        },
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedback: {
            type: String,
        },
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedbackInvalid: {
            type: String,
        },
        /**
         * Provide valuable, actionable invalid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        feedbackValid: {
            type: String,
        },
        /**
         * Set component validation state to invalid.
         */
        invalid: Boolean,
        /**
         * Display validation feedback in a styled tooltip.
         *
         * @since 4.3.0
         */
        tooltipFeedback: Boolean,
        /**
         * Set component validation state to valid.
         */
        valid: Boolean,
    },
    setup(props, { slots }) {
        return () => [
            props.feedback &&
                (props.valid || props.invalid) &&
                h$1(CFormFeedback, {
                    ...(props.invalid && { id: props.describedby }),
                    invalid: props.invalid,
                    tooltip: props.tooltipFeedback,
                    valid: props.valid,
                }, {
                    default: () => (slots.feedback && slots.feedback()) || props.feedback,
                }),
            (props.feedbackInvalid || slots.feedbackInvalid) &&
                h$1(CFormFeedback, {
                    id: props.describedby,
                    invalid: true,
                    tooltip: props.tooltipFeedback,
                }, {
                    default: () => (slots.feedbackInvalid && slots.feedbackInvalid()) || props.feedbackInvalid,
                }),
            (props.feedbackValid || slots.feedbackValid) &&
                h$1(CFormFeedback, {
                    tooltip: props.tooltipFeedback,
                    valid: true,
                }, {
                    default: () => (slots.feedbackValid && slots.feedbackValid()) || props.feedbackValid,
                }),
        ];
    },
});

const CFormLabel = defineComponent({
    name: 'CFormLabel',
    props: {
        /**
         * A string of all className you want to be applied to the component, and override standard className value.
         */
        customClassName: {
            type: [Array, String],
            default: undefined,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1('label', {
            class: props.customClassName ? props.customClassName : 'form-label',
        }, slots.default && slots.default());
    },
});

const CFormCheck = defineComponent({
    name: 'CFormCheck',
    inheritAttrs: false,
    props: {
        /**
         * Create button-like checkboxes and radio buttons.
         *
         * @see http://coreui.io/vue/docs/components/button.html
         */
        button: {
            type: Object,
        },
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedback: {
            type: String,
        },
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedbackInvalid: {
            type: String,
        },
        /**
         * Provide valuable, actionable invalid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        feedbackValid: {
            type: String,
        },
        /**
         * Sets hit area to the full area of the component.
         */
        hitArea: {
            type: String,
            validator: (value) => {
                // The value must match one of these strings
                return ['full'].includes(value);
            },
        },
        /**
         * The id global attribute defines an identifier (ID) that must be unique in the whole document.
         */
        id: {
            type: String,
        },
        /**
         * Input Checkbox indeterminate Property
         */
        indeterminate: Boolean,
        /**
         * Group checkboxes or radios on the same horizontal row by adding.
         */
        inline: {
            type: Boolean,
        },
        /**
         * Set component validation state to invalid.
         */
        invalid: Boolean,
        /**
         * The element represents a caption for a component.
         */
        label: {
            type: String,
        },
        /**
         * The default name for a value passed using v-model.
         */
        modelValue: {
            type: [Boolean, String],
            value: undefined,
        },
        /**
         * Display validation feedback in a styled tooltip.
         *
         * @since 4.3.0
         */
        tooltipFeedback: Boolean,
        /**
         * Specifies the type of component.
         *
         * @values 'checkbox', 'radio'
         */
        type: {
            type: String,
            default: 'checkbox',
        },
        /**
         * Set component validation state to valid.
         */
        valid: {
            type: Boolean,
        },
    },
    emits: [
        /**
         * Event occurs when the checked value has been changed.
         */
        'change',
        /**
         * Emit the new value whenever there’s a change event.
         */
        'update:modelValue',
    ],
    setup(props, { attrs, emit, slots }) {
        const handleChange = (event) => {
            const target = event.target;
            emit('change', event);
            emit('update:modelValue', target.checked);
        };
        const className = [
            'form-check',
            {
                'form-check-inline': props.inline,
                'is-invalid': props.invalid,
                'is-valid': props.valid,
            },
            attrs.class,
        ];
        const inputClassName = [
            props.button ? 'btn-check' : 'form-check-input',
            {
                'is-invalid': props.invalid,
                'is-valid': props.valid,
                'me-2': props.hitArea,
            },
        ];
        const formControl = () => {
            return h$1('input', {
                ...attrs,
                ...(props.modelValue && { checked: props.modelValue }),
                class: inputClassName,
                id: props.id,
                indeterminate: props.indeterminate,
                onChange: (event) => handleChange(event),
                type: props.type,
            });
        };
        const formLabel = () => props.button
            ? h$1(CButton, {
                component: 'label',
                ...props.button,
                ...(props.id && { for: props.id }),
            }, {
                default: () => (slots.label && slots.label()) || props.label,
            })
            : h$1(CFormLabel, { class: 'form-check-label', ...(props.id && { for: props.id }) }, {
                default: () => (slots.label && slots.label()) || props.label,
            });
        const formValidation = () => {
            return h$1(CFormControlValidation, {
                describedby: attrs['aria-describedby'],
                feedback: props.feedback,
                feedbackInvalid: props.feedbackInvalid,
                feedbackValid: props.feedbackValid,
                invalid: props.invalid,
                tooltipFeedback: props.tooltipFeedback,
                valid: props.valid,
            });
        };
        return () => props.button
            ? [formControl(), (slots.label || props.label) && formLabel(), formValidation()]
            : props.label
                ? props.hitArea
                    ? [
                        h$1(CFormLabel, {
                            customClassName: className,
                            ...(props.id && { for: props.id }),
                        }, [formControl(), props.label]),
                        formValidation(),
                    ]
                    : h$1('div', {
                        class: className,
                    }, [formControl(), props.label && formLabel(), formValidation()])
                : formControl();
    },
});

const CFormFloating = defineComponent({
    name: 'CFormFloating',
    setup(_, { slots }) {
        return () => h$1('div', {
            class: 'form-floating',
        }, slots.default && slots.default());
    },
});

const CFormText = defineComponent({
    name: 'CFormText',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'div',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'form-text' }, slots.default && slots.default());
    },
});

const CFormControlWrapper = defineComponent({
    name: 'CFormControlWrapper',
    inheritAttrs: false,
    props: {
        ...CFormControlValidation.props,
        /**
         * Provide valuable, actionable valid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        floatingLabel: {
            type: String,
        },
        /**
         * @ignore
         */
        id: {
            type: String,
        },
        /**
         * Add a caption for a component.
         *
         * @since 4.3.0
         */
        label: {
            type: String,
        },
        /**
         * Add helper text to the component.
         *
         * @since 4.3.0
         */
        text: {
            type: String,
        },
    },
    setup(props, { slots }) {
        const formControlValidation = () => h$1(CFormControlValidation, {
            describedby: props.describedby,
            feedback: props.feedback,
            feedbackInvalid: props.feedbackInvalid,
            feedbackValid: props.feedbackValid,
            floatingLabel: props.floatingLabel,
            invalid: props.invalid,
            tooltipFeedback: props.tooltipFeedback,
            valid: props.valid,
        }, {
            ...(slots.feedback && { feedback: () => slots.feedback && slots.feedback() }),
            ...(slots.feedbackInvalid && {
                feedbackInvalid: () => slots.feedbackInvalid && slots.feedbackInvalid(),
            }),
            ...(slots.feedbackValid && {
                feedbackValid: () => slots.feedbackInvalid && slots.feedbackInvalid(),
            }),
        });
        return () => props.floatingLabel
            ? h$1(CFormFloating, () => [
                slots.default && slots.default(),
                h$1(CFormLabel, {
                    for: props.id,
                }, {
                    default: () => (slots.label && slots.label()) || props.label || props.floatingLabel,
                }),
                (props.text || slots.text) &&
                    h$1(CFormText, {
                        id: props.describedby,
                    }, {
                        default: () => (slots.text && slots.text()) || props.text,
                    }),
                formControlValidation(),
            ])
            : [
                (props.label || slots.label) &&
                    h$1(CFormLabel, {
                        for: props.id,
                    }, {
                        default: () => (slots.label && slots.label()) || props.label,
                    }),
                slots.default && slots.default(),
                (props.text || slots.text) &&
                    h$1(CFormText, {
                        id: props.describedby,
                    }, {
                        default: () => (slots.text && slots.text()) || props.text,
                    }),
                formControlValidation(),
            ];
    },
});

const File = typeof window !== 'undefined' ? window.File : class File extends Object {
};
const CFormInput = defineComponent({
    name: 'CFormInput',
    props: {
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        // Inherited Props from CFormControlWrapper
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedback: {
            type: String,
        },
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedbackInvalid: {
            type: String,
        },
        /**
         * Provide valuable, actionable invalid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        feedbackValid: {
            type: String,
        },
        /**
         * Provide valuable, actionable valid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        floatingLabel: {
            type: String,
        },
        /**
         * The id global attribute defines an identifier (ID) that must be unique in the whole document.
         */
        id: {
            type: String,
        },
        /**
         * Set component validation state to invalid.
         */
        invalid: Boolean,
        /**
         * Add a caption for a component.
         *
         * @since 4.3.0
         */
        label: {
            type: String,
        },
        /**
         * The default name for a value passed using v-model.
         */
        modelValue: {
            type: [File, Number, String],
            default: undefined,
        },
        /**
         * Render the component styled as plain text. Removes the default form field styling and preserve the correct margin and padding. Recommend to use only along side `readonly`.
         */
        plainText: {
            type: Boolean,
            required: false,
        },
        /**
         * Toggle the readonly state for the component.
         */
        readonly: {
            type: Boolean,
            required: false,
        },
        /**
         * Size the component small or large.
         *
         * @values 'sm' | 'lg'
         */
        size: {
            type: String,
            validator: (value) => {
                return ['sm', 'lg'].includes(value);
            },
        },
        /**
         * Add helper text to the component.
         *
         * @since 4.3.0
         */
        text: {
            type: String,
        },
        /**
         * Display validation feedback in a styled tooltip.
         *
         * @since 4.3.0
         */
        tooltipFeedback: Boolean,
        /**
         * Specifies the type of component.
         *
         * @values 'color' | 'file' | 'text' | string
         */
        type: {
            type: String,
            default: 'text',
        },
        /**
         * Set component validation state to valid.
         */
        valid: Boolean,
    },
    emits: [
        /**
         * Event occurs when the element loses focus, after the content has been changed.
         */
        'change',
        /**
         * Event occurs immediately after the value of a component has changed.
         */
        'input',
        /**
         * Emit the new value whenever there’s an input or change event.
         */
        'update:modelValue',
    ],
    setup(props, { attrs, emit, slots }) {
        const handleChange = (event) => {
            const target = event.target;
            emit('change', event);
            emit('update:modelValue', target.value);
        };
        const handleInput = (event) => {
            const target = event.target;
            emit('input', event);
            emit('update:modelValue', target.value);
        };
        return () => h$1(CFormControlWrapper, {
            describedby: attrs['aria-describedby'],
            feedback: props.feedback,
            feedbackInvalid: props.feedbackInvalid,
            feedbackValid: props.feedbackValid,
            floatingLabel: props.floatingLabel,
            id: props.id,
            invalid: props.invalid,
            label: props.label,
            text: props.text,
            tooltipFeedback: props.tooltipFeedback,
            valid: props.valid,
        }, {
            default: () => h$1('input', {
                id: props.id,
                ...attrs,
                class: [
                    props.plainText ? 'form-control-plaintext' : 'form-control',
                    {
                        'form-control-color': props.type === 'color',
                        [`form-control-${props.size}`]: props.size,
                        'is-invalid': props.invalid,
                        'is-valid': props.valid,
                    },
                    attrs.class,
                ],
                disabled: props.disabled,
                onChange: (event) => handleChange(event),
                onInput: (event) => handleInput(event),
                readonly: props.readonly,
                type: props.type,
                ...((props.modelValue || props.modelValue === 0) && { value: props.modelValue })
            }, slots.default && slots.default()),
            ...(slots.feedback && { feedback: () => slots.feedback && slots.feedback() }),
            ...(slots.feedbackInvalid && {
                feedbackInvalid: () => slots.feedbackInvalid && slots.feedbackInvalid(),
            }),
            ...(slots.feedbackValid && {
                feedbackValid: () => slots.feedbackInvalid && slots.feedbackInvalid(),
            }),
            ...(slots.label && { label: () => slots.label && slots.label() }),
            ...(slots.text && { text: () => slots.text && slots.text() }),
        });
    },
});

const CFormRange = defineComponent({
    name: 'CFormRange',
    props: {
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            default: undefined,
        },
        /**
         * Add a caption for a component.
         *
         * @since 4.3.0
         */
        label: {
            type: String,
        },
        /**
         * Specifies the maximum value for the component.
         */
        max: {
            type: Number,
            default: undefined,
        },
        /**
         * Specifies the minimum value for the component.
         */
        min: {
            type: Number,
            default: undefined,
        },
        /**
         * The default name for a value passed using v-model.
         */
        modelValue: {
            type: String,
            value: undefined,
        },
        /**
         * Toggle the readonly state for the component.
         */
        readonly: {
            type: Boolean,
        },
        /**
         * Specifies the interval between legal numbers in the component.
         */
        steps: {
            type: Number,
            default: undefined,
        },
        /**
         * The `value` attribute of component.
         *
         * @controllable onChange
         * */
        value: {
            type: Number,
            default: undefined,
        },
    },
    emits: [
        /**
         * Event occurs when the value has been changed.
         */
        'change',
        /**
         * Emit the new value whenever there’s a change event.
         */
        'update:modelValue',
    ],
    setup(props, { attrs, emit, slots }) {
        const handleChange = (event) => {
            const target = event.target;
            emit('change', event);
            emit('update:modelValue', target.value);
        };
        return () => [
            props.label &&
                h$1(CFormLabel, {
                    for: attrs.id,
                }, {
                    default: () => (slots.label && slots.label()) || props.label,
                }),
            h$1('input', {
                ...attrs,
                class: 'form-range',
                disabled: props.disabled,
                max: props.max,
                min: props.min,
                onChange: (event) => handleChange(event),
                readonly: props.readonly,
                steps: props.steps,
                type: 'range',
                value: props.modelValue,
            }, slots.default && slots.default()),
        ];
    },
});

const CFormSelect = defineComponent({
    name: 'CFormSelect',
    props: {
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedback: {
            type: String,
        },
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedbackInvalid: {
            type: String,
        },
        /**
         * Provide valuable, actionable invalid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        feedbackValid: {
            type: String,
        },
        /**
         * Provide valuable, actionable valid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        floatingLabel: {
            type: String,
        },
        /**
         * Specifies the number of visible options in a drop-down list.
         */
        htmlSize: {
            type: Number,
            default: undefined,
            required: false,
        },
        /**
         * The id global attribute defines an identifier (ID) that must be unique in the whole document.
         */
        id: {
            type: String,
        },
        /**
         * Set component validation state to invalid.
         */
        invalid: Boolean,
        /**
         * Add a caption for a component.
         *
         * @since 4.3.0
         */
        label: {
            type: String,
        },
        /**
         * The default name for a value passed using v-model.
         */
        modelValue: {
            type: [String, Array],
            default: undefined,
            require: false,
        },
        multiple: {
            type: Boolean,
            required: false,
        },
        /**
         * Options list of the select component. Available keys: `label`, `value`, `disabled`.
         * Examples:
         * - `:options="[{ value: 'js', label: 'JavaScript' }, { value: 'html', label: 'HTML', disabled: true }]"`
         * - `:options="['js', 'html']"`
         */
        options: {
            type: Array,
            default: undefined,
            required: false,
        },
        /**
         * Size the component small or large.
         *
         * @values 'sm' | 'lg'
         */
        size: {
            type: String,
            default: undefined,
            require: false,
            validator: (value) => {
                return ['sm', 'lg'].includes(value);
            },
        },
        /**
         * Add helper text to the component.
         *
         * @since 4.3.0
         */
        text: {
            type: String,
        },
        /**
         * Display validation feedback in a styled tooltip.
         *
         * @since 4.3.0
         */
        tooltipFeedback: Boolean,
        /**
         * Set component validation state to valid.
         */
        valid: Boolean,
    },
    emits: [
        /**
         * Event occurs when when a user changes the selected option of a `<select>` element.
         */
        'change',
        /**
         * Emit the new value whenever there’s a change event.
         */
        'update:modelValue',
    ],
    setup(props, { attrs, emit, slots }) {
        const handleChange = (event) => {
            const target = event.target;
            const selected = Array.from(target.options)
                .filter((option) => option.selected)
                .map((option) => option.value);
            emit('change', event);
            emit('update:modelValue', target.multiple ? selected : selected[0]);
        };
        return () => h$1(CFormControlWrapper, {
            describedby: attrs['aria-describedby'],
            feedback: props.feedback,
            feedbackInvalid: props.feedbackInvalid,
            feedbackValid: props.feedbackValid,
            floatingLabel: props.floatingLabel,
            id: props.id,
            invalid: props.invalid,
            label: props.label,
            text: props.text,
            tooltipFeedback: props.tooltipFeedback,
            valid: props.valid,
        }, {
            default: () => h$1('select', {
                id: props.id,
                ...attrs,
                class: [
                    'form-select',
                    {
                        [`form-select-${props.size}`]: props.size,
                        'is-invalid': props.invalid,
                        'is-valid': props.valid,
                    },
                    attrs.class,
                ],
                multiple: props.multiple,
                onChange: (event) => handleChange(event),
                size: props.htmlSize,
                ...(props.modelValue && !props.multiple && { value: props.modelValue }),
            }, props.options
                ? props.options.map((option) => {
                    return h$1('option', {
                        ...(typeof option === 'object' && {
                            ...(option.disabled && { disabled: option.disabled }),
                            ...(option.selected && { selected: option.selected }),
                            ...(option.value && {
                                value: option.value,
                                ...(props.modelValue &&
                                    props.multiple &&
                                    props.modelValue.includes(option.value) && { selected: true }),
                            }),
                        }),
                    }, typeof option === 'string' ? option : option.label);
                })
                : slots.default && slots.default()),
            ...(slots.feedback && { feedback: () => slots.feedback && slots.feedback() }),
            ...(slots.feedbackInvalid && {
                feedbackInvalid: () => slots.feedbackInvalid && slots.feedbackInvalid(),
            }),
            ...(slots.feedbackValid && {
                feedbackValid: () => slots.feedbackInvalid && slots.feedbackInvalid(),
            }),
            ...(slots.label && { label: () => slots.label && slots.label() }),
            ...(slots.text && { text: () => slots.text && slots.text() }),
        });
    },
});

const CFormSwitch = defineComponent({
    name: 'CFormSwitch',
    inheritAttrs: false,
    props: {
        /**
         * The id global attribute defines an identifier (ID) that must be unique in the whole document
         */
        id: {
            type: String,
        },
        /**
         * Set component validation state to invalid.
         */
        invalid: {
            type: Boolean,
        },
        /**
         * The element represents a caption for a component.
         */
        label: {
            type: String,
        },
        /**
         * The default name for a value passed using v-model.
         */
        modelValue: {
            type: [Boolean, String],
            value: undefined,
        },
        /**
         * Size the component large or extra large. Works only with `switch`.
         *
         * @values 'lg' | 'xl'
         */
        size: {
            type: String,
            validator: (value) => {
                return ['lg', 'xl'].includes(value);
            },
        },
        /**
         * Specifies the type of component.
         *
         * @values 'checkbox', 'radio'
         */
        type: {
            type: String,
            default: 'checkbox',
        },
        /**
         * Set component validation state to valid.
         */
        valid: {
            type: Boolean,
        },
    },
    emits: [
        /**
         * Event occurs when the checked value has been changed.
         */
        'change',
        /**
         * Emit the new value whenever there’s a change event.
         */
        'update:modelValue',
    ],
    setup(props, { attrs, emit }) {
        const handleChange = (event) => {
            const target = event.target;
            emit('change', event);
            emit('update:modelValue', target.checked);
        };
        return () => h$1('div', {
            class: [
                'form-check form-switch',
                {
                    [`form-switch-${props.size}`]: props.size,
                    'is-invalid': props.invalid,
                    'is-valid': props.valid,
                },
            ],
        }, [
            h$1('input', {
                ...attrs,
                ...(props.modelValue && { checked: props.modelValue }),
                class: [
                    'form-check-input',
                    {
                        'is-invalid': props.invalid,
                        'is-valid': props.valid,
                    },
                ],
                id: props.id,
                onChange: (event) => handleChange(event),
                type: props.type,
            }),
            props.label &&
                h$1(CFormLabel, {
                    ...(props.id && { for: props.id }),
                    class: 'form-check-label',
                }, {
                    default: () => props.label,
                }),
        ]);
    },
});

const CFormTextarea = defineComponent({
    name: 'CFormTextarea',
    props: {
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
        },
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedback: {
            type: String,
        },
        /**
         * Provide valuable, actionable feedback.
         *
         * @since 4.3.0
         */
        feedbackInvalid: {
            type: String,
        },
        /**
         * Provide valuable, actionable invalid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        feedbackValid: {
            type: String,
        },
        /**
         * Provide valuable, actionable valid feedback when using standard HTML form validation which applied two CSS pseudo-classes, `:invalid` and `:valid`.
         *
         * @since 4.3.0
         */
        floatingLabel: {
            type: String,
        },
        /**
         * The id global attribute defines an identifier (ID) that must be unique in the whole document.
         */
        id: {
            type: String,
        },
        /**
         * Set component validation state to invalid.
         */
        invalid: Boolean,
        /**
         * Add a caption for a component.
         *
         * @since 4.3.0
         */
        label: {
            type: String,
        },
        /**
         * The default name for a value passed using v-model.
         */
        modelValue: {
            type: String,
            default: undefined,
        },
        /**
         * Render the component styled as plain text. Removes the default form field styling and preserve the correct margin and padding. Recommend to use only along side `readonly`.
         */
        plainText: {
            type: Boolean,
        },
        /**
         * Toggle the readonly state for the component.
         */
        readonly: {
            type: Boolean,
        },
        /**
         * Add helper text to the component.
         *
         * @since 4.3.0
         */
        text: {
            type: String,
        },
        /**
         * Display validation feedback in a styled tooltip.
         *
         * @since 4.3.0
         */
        tooltipFeedback: Boolean,
        /**
         * Set component validation state to valid.
         */
        valid: Boolean,
    },
    emits: [
        /**
         * Event occurs when the element loses focus, after the content has been changed.
         */
        'change',
        /**
         * Event occurs immediately after the value of a component has changed.
         */
        'input',
        /**
         * Emit the new value whenever there’s an input or change event.
         */
        'update:modelValue',
    ],
    setup(props, { attrs, emit, slots }) {
        const handleChange = (event) => {
            const target = event.target;
            emit('change', event);
            emit('update:modelValue', target.value);
        };
        const handleInput = (event) => {
            const target = event.target;
            emit('input', event);
            emit('update:modelValue', target.value);
        };
        return () => h$1(CFormControlWrapper, {
            describedby: attrs['aria-describedby'],
            feedback: props.feedback,
            feedbackInvalid: props.feedbackInvalid,
            feedbackValid: props.feedbackValid,
            floatingLabel: props.floatingLabel,
            id: props.id,
            invalid: props.invalid,
            label: props.label,
            text: props.text,
            tooltipFeedback: props.tooltipFeedback,
            valid: props.valid,
        }, {
            default: () => h$1('textarea', {
                id: props.id,
                ...attrs,
                disabled: props.disabled,
                readonly: props.readonly,
                class: [
                    props.plainText ? 'form-control-plaintext' : 'form-control',
                    {
                        'is-invalid': props.invalid,
                        'is-valid': props.valid,
                    },
                ],
                onChange: (event) => handleChange(event),
                onInput: (event) => handleInput(event),
                ...(props.modelValue && { value: props.modelValue }),
            }, slots.default && slots.default()),
            ...(slots.feedback && { feedback: () => slots.feedback && slots.feedback() }),
            ...(slots.feedbackInvalid && {
                feedbackInvalid: () => slots.feedbackInvalid && slots.feedbackInvalid(),
            }),
            ...(slots.feedbackValid && {
                feedbackValid: () => slots.feedbackInvalid && slots.feedbackInvalid(),
            }),
            ...(slots.label && { label: () => slots.label && slots.label() }),
            ...(slots.text && { text: () => slots.text && slots.text() }),
        });
    },
});

const CInputGroup = defineComponent({
    name: 'CInputGroup',
    props: {
        /**
         * Size the component small or large.
         *
         * @values 'sm', 'lg'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['sm', 'lg'].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1('div', {
            class: [
                'input-group',
                {
                    [`input-group-${props.size}`]: props.size,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CInputGroupText = defineComponent({
    name: 'CInputGroupText',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'span',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'input-group-text' }, slots.default && slots.default());
    },
});

const CFormPlugin = {
    install: (app) => {
        app.component(CForm.name, CForm);
        app.component(CFormCheck.name, CFormCheck);
        // app.component(CFormControl.name, CFormControl)
        app.component(CFormFeedback.name, CFormFeedback);
        app.component(CFormFloating.name, CFormFloating);
        app.component(CFormInput.name, CFormInput);
        app.component(CFormLabel.name, CFormLabel);
        app.component(CFormRange.name, CFormRange);
        app.component(CFormSelect.name, CFormSelect);
        app.component(CFormSwitch.name, CFormSwitch);
        app.component(CFormText.name, CFormText);
        app.component(CFormTextarea.name, CFormTextarea);
        app.component(CInputGroup.name, CInputGroup);
        app.component(CInputGroupText.name, CInputGroupText);
    },
};

const BREAKPOINTS$4 = [
    'xxl',
    'xl',
    'lg',
    'md',
    'sm',
    'xs',
];
const CCol = defineComponent({
    name: 'CCol',
    props: {
        /**
         * The number of columns/offset/order on extra small devices (<576px).
         *
         * @values 'auto' | number | string | boolean | { span: 'auto' | number | string | boolean } | { offset: number | string } | { order: 'first' | 'last' | number | string }
         */
        xs: {
            type: [Boolean, Number, String, Object],
            default: undefined,
            require: false,
        },
        /**
         * The number of columns/offset/order on small devices (<768px).
         *
         * @values 'auto' | number | string | boolean | { span: 'auto' | number | string | boolean } | { offset: number | string } | { order: 'first' | 'last' | number | string }
         */
        sm: {
            type: [Boolean, Number, String, Object],
            default: undefined,
            require: false,
        },
        /**
         * The number of columns/offset/order on medium devices (<992px).
         *
         * @values 'auto' | number | string | boolean | { span: 'auto' | number | string | boolean } | { offset: number | string } | { order: 'first' | 'last' | number | string }
         */
        md: {
            type: [Boolean, Number, String, Object],
            default: undefined,
            require: false,
        },
        /**
         * The number of columns/offset/order on large devices (<1200px).
         *
         * @values 'auto' | number | string | boolean | { span: 'auto' | number | string | boolean } | { offset: number | string } | { order: 'first' | 'last' | number | string }
         */
        lg: {
            type: [Boolean, Number, String, Object],
            default: undefined,
            require: false,
        },
        /**
         * The number of columns/offset/order on X-Large devices (<1400px).
         *
         * @values 'auto' | number | string | boolean | { span: 'auto' | number | string | boolean } | { offset: number | string } | { order: 'first' | 'last' | number | string }
         */
        xl: {
            type: [Boolean, Number, String, Object],
            default: undefined,
            require: false,
        },
        /**
         * The number of columns/offset/order on XX-Large devices (≥1400px).
         *
         * @values 'auto' | number | string | boolean | { span: 'auto' | number | string | boolean } | { offset: number | string } | { order: 'first' | 'last' | number | string }
         */
        xxl: {
            type: [Boolean, Number, String, Object],
            default: undefined,
            require: false,
        },
    },
    setup(props, { slots }) {
        const repsonsiveClassNames = [];
        BREAKPOINTS$4.forEach((bp) => {
            const breakpoint = props[bp];
            const infix = bp === 'xs' ? '' : `-${bp}`;
            if (breakpoint) {
                if (typeof breakpoint === 'number' || typeof breakpoint === 'string') {
                    repsonsiveClassNames.push(`col${infix}-${breakpoint}`);
                }
                if (typeof breakpoint === 'boolean') {
                    repsonsiveClassNames.push(`col${infix}`);
                }
            }
            if (breakpoint && typeof breakpoint === 'object') {
                if (typeof breakpoint.span === 'number' || typeof breakpoint.span === 'string') {
                    repsonsiveClassNames.push(`col${infix}-${breakpoint.span}`);
                }
                if (typeof breakpoint.span === 'boolean') {
                    repsonsiveClassNames.push(`col${infix}`);
                }
                if (typeof breakpoint.order === 'number' || typeof breakpoint.order === 'string') {
                    repsonsiveClassNames.push(`order${infix}-${breakpoint.order}`);
                }
                if (typeof breakpoint.offset === 'number') {
                    repsonsiveClassNames.push(`offset${infix}-${breakpoint.offset}`);
                }
            }
        });
        return () => h$1('div', {
            class: [repsonsiveClassNames.length ? repsonsiveClassNames : 'col'],
        }, slots.default && slots.default());
    },
});

const BREAKPOINTS$3 = [
    'xxl',
    'xl',
    'lg',
    'md',
    'sm',
    'fluid',
];
const CContainer = defineComponent({
    name: 'CContainer',
    props: {
        /**
         * Set container 100% wide until small breakpoint.
         */
        sm: {
            type: Boolean,
            required: false,
        },
        /**
         * Set container 100% wide until medium breakpoint.
         */
        md: {
            type: Boolean,
            required: false,
        },
        /**
         * Set container 100% wide until large breakpoint.
         */
        lg: {
            type: Boolean,
            required: false,
        },
        /**
         * Set container 100% wide until X-large breakpoint.
         */
        xl: {
            type: Boolean,
            required: false,
        },
        /**
         * Set container 100% wide until XX-large breakpoint.
         */
        xxl: {
            type: Boolean,
            required: false,
        },
        /**
         * Set container 100% wide, spanning the entire width of the viewport.
         */
        fluid: {
            type: Boolean,
            required: false,
        },
    },
    setup(props, { slots }) {
        const repsonsiveClassNames = [];
        BREAKPOINTS$3.forEach((bp) => {
            const breakpoint = props[bp];
            breakpoint && repsonsiveClassNames.push(`container-${bp}`);
        });
        return () => h$1('div', {
            class: [repsonsiveClassNames.length ? repsonsiveClassNames : 'container'],
        }, slots.default && slots.default());
    },
});

const BREAKPOINTS$2 = [
    'xxl',
    'xl',
    'lg',
    'md',
    'sm',
    'xs',
];
const CRow = defineComponent({
    name: 'CRow',
    props: {
        /**
         * The number of columns/offset/order on extra small devices (<576px).
         *
         * @values { cols: 'auto' | number | string } | { gutter: number | string } | { gutterX: number | string } | { gutterY: number | string }
         */
        xs: {
            type: Object,
            default: undefined,
            required: false,
        },
        /**
         * The number of columns/offset/order on small devices (<768px).
         *
         * @values { cols: 'auto' | number | string } | { gutter: number | string } | { gutterX: number | string } | { gutterY: number | string }
         */
        sm: {
            type: Object,
            default: undefined,
            required: false,
            /**
             * The number of columns/offset/order on medium devices (<992px).
             *
             * @values { cols: 'auto' | number | string } | { gutter: number | string } | { gutterX: number | string } | { gutterY: number | string }
             */
        },
        md: {
            type: Object,
            default: undefined,
            required: false,
        },
        /**
         * The number of columns/offset/order on large devices (<1200px).
         *
         * @values { cols: 'auto' | number | string } | { gutter: number | string } | { gutterX: number | string } | { gutterY: number | string }
         */
        lg: {
            type: Object,
            default: undefined,
            required: false,
        },
        /**
         * The number of columns/offset/order on X-Large devices (<1400px).
         *
         * @values { cols: 'auto' | number | string } | { gutter: number | string } | { gutterX: number | string } | { gutterY: number | string }
         */
        xl: {
            type: Object,
            default: undefined,
            required: false,
        },
        /**
         * The number of columns/offset/order on XX-Large devices (≥1400px).
         *
         * @values { cols: 'auto' | number | string } | { gutter: number | string } | { gutterX: number | string } | { gutterY: number | string }
         */
        xxl: {
            type: Object,
            default: undefined,
            required: false,
        },
    },
    setup(props, { slots }) {
        const repsonsiveClassNames = [];
        BREAKPOINTS$2.forEach((bp) => {
            const breakpoint = props[bp];
            const infix = bp === 'xs' ? '' : `-${bp}`;
            if (typeof breakpoint === 'object') {
                if (breakpoint.cols) {
                    repsonsiveClassNames.push(`row-cols${infix}-${breakpoint.cols}`);
                }
                if (typeof breakpoint.gutter === 'number') {
                    repsonsiveClassNames.push(`g${infix}-${breakpoint.gutter}`);
                }
                if (typeof breakpoint.gutterX === 'number') {
                    repsonsiveClassNames.push(`gx${infix}-${breakpoint.gutterX}`);
                }
                if (typeof breakpoint.gutterY === 'number') {
                    repsonsiveClassNames.push(`gy${infix}-${breakpoint.gutterY}`);
                }
            }
        });
        return () => h$1('div', {
            class: ['row', repsonsiveClassNames],
        }, slots.default && slots.default());
    },
});

const CGridPlugin = {
    install: (app) => {
        app.component(CCol.name, CCol);
        app.component(CContainer.name, CContainer);
        app.component(CRow.name, CRow);
    },
};

const CHeader = defineComponent({
    name: 'CHeader',
    props: {
        /**
         * Defines optional container wrapping children elements.
         *
         * @values boolean, 'sm', 'md', 'lg', 'xl', 'xxl', 'fluid'
         */
        container: {
            type: [Boolean, String],
            default: undefined,
            required: false,
            validator: (value) => {
                return (typeof value === 'boolean' || ['sm', 'md', 'lg', 'xl', 'xxl', 'fluid'].includes(value));
            },
        },
        /**
         * Place header in non-static positions.
         *
         * @values 'fixed', 'sticky'
         */
        position: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['fixed', 'sticky'].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1('div', { class: ['header', { [`header-${props.position}`]: props.position }] }, props.container
            ? h$1('div', { class: `container${props.container !== true ? '-' + props.container : ''}` }, slots.default && slots.default())
            : slots.default && slots.default());
    },
});

const CHeaderBrand = defineComponent({
    name: 'CHeaderBrand',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'a',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'header-brand' }, slots.default && slots.default());
    },
});

const CHeaderDivider = defineComponent({
    name: 'CHeaderDivider',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'header-divider' }, slots.default && slots.default());
    },
});

const CHeaderNav = defineComponent({
    name: 'CHeaderNav',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'ul',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: 'header-nav',
            role: 'navigation',
        }, slots.default && slots.default());
    },
});

const CHeaderText = defineComponent({
    name: 'CHeaderText',
    setup(_, { slots }) {
        return () => h$1('span', { class: 'header-text' }, slots.default && slots.default());
    },
});

const CHeaderToggler = defineComponent({
    name: 'CHeaderToggler',
    setup(_, { slots }) {
        return () => h$1('button', {
            class: 'header-toggler',
            type: 'button',
            'aria-label': 'Toggle navigation',
        }, slots.default ? slots.default() : h$1('span', { class: ['header-toggler-icon'] }));
    },
});

const CHeaderPlugin = {
    install: (app) => {
        app.component(CHeader.name, CHeader);
        app.component(CHeaderBrand.name, CHeaderBrand);
        app.component(CHeaderDivider.name, CHeaderDivider);
        app.component(CHeaderNav.name, CHeaderNav);
        app.component(CHeaderText.name, CHeaderText);
        app.component(CHeaderToggler.name, CHeaderToggler);
    },
};

const CImage = defineComponent({
    name: 'CImage',
    props: {
        /**
         * Set the horizontal aligment.
         *
         * @values 'start', 'center', 'end'
         */
        align: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['start', 'center', 'end'].includes(value);
            },
        },
        /**
         * Make image responsive.
         */
        fluid: {
            type: Boolean,
            required: false,
        },
        /**
         * Make image rounded.
         */
        rounded: {
            type: Boolean,
            required: false,
        },
        /**
         * Give an image a rounded 1px border appearance.
         */
        thumbnail: {
            type: Boolean,
            required: false,
        },
    },
    setup(props) {
        return () => h$1('img', {
            class: [
                {
                    [`float-${props.align}`]: props.align && (props.align === 'start' || props.align === 'end'),
                    'd-block mx-auto': props.align && props.align === 'center',
                    'img-fluid': props.fluid,
                    rounded: props.rounded,
                    'img-thumbnail': props.thumbnail,
                },
            ],
        });
    },
});

const CImagePlugin = {
    install: (app) => {
        app.component(CImage.name, CImage);
    },
};

const CListGroup = defineComponent({
    name: 'CListGroup',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'ul',
        },
        /**
         * Remove some borders and rounded corners to render list group items edge-to-edge in a parent component (e.g., `<CCard>`)
         */
        flush: {
            type: Boolean,
            required: false,
        },
        /**
         * Specify a layout type.
         *
         * @values 'horizontal', 'horizontal-sm', 'horizontal-md', 'horizontal-lg', 'horizontal-xl', 'horizontal-xxl',
         */
        layout: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return [
                    'horizontal',
                    'horizontal-sm',
                    'horizontal-md',
                    'horizontal-lg',
                    'horizontal-xl',
                    'horizontal-xxl',
                ].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: [
                'list-group',
                {
                    'list-group-flush': props.flush,
                    [`list-group-${props.layout}`]: props.layout,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CListGroupItem = defineComponent({
    name: 'CListGroupItem',
    props: {
        /**
         * Toggle the active state for the component.
         */
        active: {
            type: Boolean,
            required: false,
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'li',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: [
                'list-group-item',
                {
                    [`list-group-item-${props.color}`]: props.color,
                    'list-group-item-action': props.component === 'a' || props.component === 'button',
                    [`active`]: props.active,
                    [`disabled`]: props.disabled,
                },
            ],
            ...((props.component === 'a' || props.component === 'button') && {
                active: props.active,
                disabled: props.disabled,
            }),
            ...(props.active && { 'aria-current': true }),
            ...(props.disabled && { 'aria-disabled': true }),
        }, slots.default && slots.default());
    },
});

const CListGroupPlugin = {
    install: (app) => {
        app.component(CListGroup.name, CListGroup);
        app.component(CListGroupItem.name, CListGroupItem);
    },
};

const CModal = defineComponent({
    name: 'CModal',
    inheritAttrs: false,
    props: {
        /**
         * Align the modal in the center or top of the screen.
         *
         * @values 'top', 'center'
         */
        alignment: {
            default: 'top',
            validator: (value) => {
                return ['top', 'center'].includes(value);
            },
        },
        /**
         * Apply a backdrop on body while offcanvas is open.
         *
         * @values boolean | 'static'
         */
        backdrop: {
            type: [Boolean, String],
            default: true,
            validator: (value) => {
                if (typeof value == 'string') {
                    return ['static'].includes(value);
                }
                if (typeof value == 'boolean') {
                    return true;
                }
                return false;
            },
        },
        /**
         * A string of all className you want applied to the modal content component.
         */
        contentClassName: {
            type: String,
            default: undefined,
        },
        /**
         * Set modal to covers the entire user viewport
         *
         * @values boolean, 'sm', 'md', 'lg', 'xl', 'xxl'
         */
        fullscreen: {
            type: [Boolean, String],
            default: undefined,
            validator: (value) => {
                if (typeof value == 'string') {
                    return ['sm', 'md', 'lg', 'xl', 'xxl'].includes(value);
                }
                if (typeof value == 'boolean') {
                    return true;
                }
                return false;
            },
        },
        /**
         * Closes the modal when escape key is pressed.
         */
        keyboard: {
            type: Boolean,
            default: true,
        },
        /**
         * Create a scrollable modal that allows scrolling the modal body.
         */
        scrollable: {
            type: Boolean,
        },
        /**
         * Size the component small, large, or extra large.
         *
         * @values 'sm', 'lg', 'xl'
         */
        size: {
            type: String,
            default: undefined,
            validator: (value) => {
                return ['sm', 'lg', 'xl'].includes(value);
            },
        },
        /**
         * Remove animation to create modal that simply appear rather than fade in to view.
         */
        transition: {
            type: Boolean,
            default: true,
        },
        /**
         * By default the component is unmounted after close animation, if you want to keep the component mounted set this property to false.
         */
        unmountOnClose: {
            type: Boolean,
            default: true,
        },
        /**
         * Toggle the visibility of alert component.
         */
        visible: Boolean,
    },
    emits: [
        /**
         * Callback fired when the component requests to be closed.
         */
        'close',
        /**
         * Callback fired when the component requests to be closed.
         */
        'close-prevented',
        /**
         * Callback fired when the modal is shown, its backdrop is static and a click outside the modal or an escape key press is performed with the keyboard option set to false.
         */
        'show',
    ],
    setup(props, { slots, attrs, emit }) {
        const modalRef = ref();
        const modalContentRef = ref();
        const visible = ref(props.visible);
        watch(() => props.visible, () => {
            visible.value = props.visible;
        });
        const handleEnter = (el, done) => {
            executeAfterTransition(() => done(), el);
            document.body.classList.add('modal-open');
            document.body.style.overflow = 'hidden';
            document.body.style.paddingRight = '0px';
            el.style.display = 'block';
            setTimeout(() => {
                el.classList.add('show');
            }, 1);
            emit('show');
        };
        const handleAfterEnter = () => {
            window.addEventListener('mousedown', handleMouseDown);
            window.addEventListener('keyup', handleKeyUp);
        };
        const handleLeave = (el, done) => {
            executeAfterTransition(() => done(), el);
            document.body.classList.remove('modal-open');
            document.body.style.removeProperty('overflow');
            document.body.style.removeProperty('padding-right');
            if (document.body.className === '') {
                document.body.removeAttribute('class');
            }
            el.classList.remove('show');
        };
        const handleAfterLeave = (el) => {
            window.removeEventListener('mousedown', handleMouseDown);
            window.removeEventListener('keyup', handleKeyUp);
            el.style.display = 'none';
        };
        const handleDismiss = () => {
            emit('close');
            visible.value = false;
        };
        const handleKeyUp = (event) => {
            if (modalContentRef.value && !modalContentRef.value.contains(event.target)) {
                if (props.backdrop !== 'static' && event.key === 'Escape' && props.keyboard) {
                    handleDismiss();
                }
                if (props.backdrop === 'static') {
                    modalRef.value.classList.add('modal-static');
                    emit('close-prevented');
                    setTimeout(() => {
                        modalRef.value.classList.remove('modal-static');
                    }, 300);
                }
            }
        };
        const handleMouseDown = (event) => {
            window.addEventListener('mouseup', () => handleMouseUp(event), { once: true });
        };
        const handleMouseUp = (event) => {
            if (modalContentRef.value && !modalContentRef.value.contains(event.target)) {
                if (props.backdrop !== 'static') {
                    handleDismiss();
                }
                if (props.backdrop === 'static') {
                    modalRef.value.classList.add('modal-static');
                    setTimeout(() => {
                        modalRef.value.classList.remove('modal-static');
                    }, 300);
                }
            }
        };
        provide('handleDismiss', handleDismiss);
        const modal = () => h$1('div', {
            class: [
                'modal',
                {
                    ['fade']: props.transition,
                },
                attrs.class,
            ],
            ref: modalRef,
        }, h$1('div', {
            class: [
                'modal-dialog',
                {
                    'modal-dialog-centered': props.alignment === 'center',
                    [`modal-fullscreen-${props.fullscreen}-down`]: props.fullscreen && typeof props.fullscreen === 'string',
                    'modal-fullscreen': props.fullscreen && typeof props.fullscreen === 'boolean',
                    ['modal-dialog-scrollable']: props.scrollable,
                    [`modal-${props.size}`]: props.size,
                },
            ],
            role: 'dialog',
        }, h$1('div', { class: ['modal-content', props.contentClassName], ref: modalContentRef }, slots.default && slots.default())));
        return () => [
            h$1(Transition, {
                css: false,
                onEnter: (el, done) => handleEnter(el, done),
                onAfterEnter: () => handleAfterEnter(),
                onLeave: (el, done) => handleLeave(el, done),
                onAfterLeave: (el) => handleAfterLeave(el),
            }, () => props.unmountOnClose
                ? visible.value && modal()
                : withDirectives(modal(), [[vShow, visible.value]])),
            props.backdrop &&
                h$1(CBackdrop, {
                    class: 'modal-backdrop',
                    visible: visible.value,
                }),
        ];
    },
});

const CModalBody = defineComponent({
    name: 'CModalBody',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'modal-body' }, slots.default && slots.default());
    },
});

const CModalFooter = defineComponent({
    name: 'CModalFooter',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'modal-footer' }, slots.default && slots.default());
    },
});

const CModalHeader = defineComponent({
    name: 'CModalHeader',
    props: {
        /**
         * Add a close button component to the header.
         */
        closeButton: {
            type: Boolean,
            required: false,
            default: true,
        },
    },
    setup(props, { slots }) {
        const handleDismiss = inject('handleDismiss');
        return () => h$1('span', { class: 'modal-header' }, [
            slots.default && slots.default(),
            props.closeButton && h$1(CCloseButton, { onClick: () => handleDismiss() }, ''),
        ]);
    },
});

const CModalTitle = defineComponent({
    name: 'CModalTitle',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'h5',
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'modal-title' }, slots.default && slots.default());
    },
});

const CModalPlugin = {
    install: (app) => {
        app.component(CModal.name, CModal);
        app.component(CModalBody.name, CModalBody);
        app.component(CModalFooter.name, CModalFooter);
        app.component(CModalHeader.name, CModalHeader);
        app.component(CModalTitle.name, CModalTitle);
    },
};

const CNav = defineComponent({
    name: 'CNav',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'ul',
        },
        /**
         * Specify a layout type for component.
         *
         * @values 'fill', 'justified'
         */
        layout: {
            type: String,
            required: false,
            default: undefined,
            validator: (value) => {
                return ['fill', 'justified'].includes(value);
            },
        },
        /**
         * Set the nav variant to tabs or pills.
         *
         * @values 'tabs', 'pills'
         */
        variant: {
            type: String,
            required: false,
            default: undefined,
            validator: (value) => {
                return ['tabs', 'pills'].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: [
                'nav',
                {
                    [`nav-${props.layout}`]: props.layout,
                    [`nav-${props.variant}`]: props.variant,
                },
            ],
            role: 'navigation',
        }, slots.default && slots.default());
    },
});

const CNavGroup = defineComponent({
    name: 'CNavGroup',
    props: {
        /**
         * Make nav group more compact by cutting all `padding` in half.
         */
        compact: Boolean,
        /**
         * Show nav group items.
         */
        visible: Boolean,
    },
    emits: ['visible-change'],
    setup(props, { slots, emit }) {
        const visible = ref();
        const navGroupRef = ref();
        const visibleGroup = ref();
        const handleVisibleChange = (visible, index) => {
            if (visible) {
                visibleGroup.value = index;
            }
            else {
                if (visibleGroup.value === index) {
                    visibleGroup.value = 0;
                }
            }
        };
        const isVisible = (index) => Boolean(visibleGroup.value === index);
        onMounted(() => {
            visible.value = props.visible;
            props.visible && navGroupRef.value.classList.add('show');
            emit('visible-change', visible.value);
        });
        watch(() => props.visible, () => {
            visible.value = props.visible;
            if (visible.value === false) {
                visibleGroup.value = undefined;
            }
        });
        watch(visible, () => {
            emit('visible-change', visible.value);
        });
        const handleTogglerClick = () => {
            visible.value = !visible.value;
            emit('visible-change', visible.value);
        };
        const handleBeforeEnter = (el) => {
            el.style.height = '0px';
            navGroupRef.value.classList.add('show');
        };
        const handleEnter = (el, done) => {
            executeAfterTransition(() => done(), el);
            el.style.height = `${el.scrollHeight}px`;
        };
        const handleAfterEnter = (el) => {
            el.style.height = 'auto';
        };
        const handleBeforeLeave = (el) => {
            el.style.height = `${el.scrollHeight}px`;
        };
        const handleLeave = (el, done) => {
            executeAfterTransition(() => done(), el);
            setTimeout(() => {
                el.style.height = '0px';
            }, 1);
        };
        const handleAfterLeave = () => {
            navGroupRef.value.classList.remove('show');
        };
        return () => h$1('li', {
            class: 'nav-group',
            ref: navGroupRef,
        }, [
            slots.togglerContent &&
                h$1('a', {
                    class: ['nav-link', 'nav-group-toggle'],
                    onClick: handleTogglerClick,
                }, slots.togglerContent && slots.togglerContent()),
            h$1(Transition, {
                css: false,
                onBeforeEnter: (el) => handleBeforeEnter(el),
                onEnter: (el, done) => handleEnter(el, done),
                onAfterEnter: (el) => handleAfterEnter(el),
                onBeforeLeave: (el) => handleBeforeLeave(el),
                onLeave: (el, done) => handleLeave(el, done),
                onAfterLeave: () => handleAfterLeave(),
            }, {
                default: () => visible.value &&
                    h$1('ul', {
                        class: [
                            'nav-group-items',
                            {
                                compact: props.compact,
                            },
                        ],
                    }, slots.default &&
                        slots.default().map((vnode, index) => {
                            // @ts-expect-error name is defined in component
                            if (vnode.type.name === 'CNavGroup') {
                                return h$1(vnode, {
                                    onVisibleChange: (visible) => handleVisibleChange(visible, index + 1),
                                    ...(visibleGroup.value && { visible: isVisible(index + 1) }),
                                });
                            }
                            return vnode;
                        })),
            }),
        ]);
    },
});

const CNavGroupItems = defineComponent({
    name: 'CNavGroupItems',
    setup(_, { slots }) {
        return () => h$1('ul', { class: 'nav-group-items' }, slots.default && slots.default());
    },
});

const CNavLink = defineComponent({
    name: 'CNavLink',
    props: {
        /**
         * Toggle the active state for the component.
         */
        active: {
            type: Boolean,
            required: false,
        },
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'a',
        },
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * @ignore
         */
        href: {
            type: String,
            default: undefined,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(CLink, {
            class: 'nav-link',
            active: props.active,
            component: props.component,
            disabled: props.disabled,
            href: props.href,
        }, {
            default: () => slots.default && slots.default(),
        });
    },
});

const CNavItem = defineComponent({
    name: 'CNavItem',
    props: {
        ...CNavLink.props,
    },
    setup(props, { slots }) {
        return () => h$1('li', {
            class: 'nav-item',
        }, props.href
            ? h$1(CNavLink, {
                active: props.active,
                component: props.component,
                disabled: props.disabled,
                href: props.href,
            }, {
                default: () => slots.default && slots.default(),
            })
            : slots.default && slots.default());
    },
});

const CNavTitle = defineComponent({
    name: 'CNavTitle',
    setup(_, { slots }) {
        return () => h$1('li', { class: 'nav-title' }, slots.default && slots.default());
    },
});

const CNavPlugin = {
    install: (app) => {
        app.component(CNav.name, CNav);
        app.component(CNavGroup.name, CNavGroup);
        app.component(CNavGroupItems.name, CNavGroupItems);
        app.component(CNavItem.name, CNavItem);
        app.component(CNavLink.name, CNavLink);
        app.component(CNavTitle.name, CNavTitle);
    },
};

const CNavbar = defineComponent({
    name: 'CNavbar',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values  'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
        /**
         * Sets if the color of text should be colored for a light or dark dark background.
         *
         * @values 'dark', 'light'
         */
        colorScheme: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['dark', 'light'].includes(value);
            },
        },
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'nav',
            required: false,
        },
        /**
         * Defines optional container wrapping children elements.
         *
         * @values boolean, 'sm', 'md', 'lg', 'xl', 'xxl', 'fluid'
         */
        container: {
            type: [Boolean, String],
            default: undefined,
            required: false,
            validator: (value) => {
                return (typeof value === 'boolean' || ['sm', 'md', 'lg', 'xl', 'xxl', 'fluid'].includes(value));
            },
        },
        /**
         * Defines the responsive breakpoint to determine when content collapses.
         *
         * @values boolean, 'sm', 'md', 'lg', 'xl', 'xxl'
         */
        expand: {
            type: [Boolean, String],
            default: undefined,
            required: false,
            validator: (value) => {
                return typeof value === 'boolean' || ['sm', 'md', 'lg', 'xl', 'xxl'].includes(value);
            },
        },
        /**
         * Place component in non-static positions.
         *
         * @values 'fixed-top', 'fixed-bottom', 'sticky-top'
         */
        placement: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['fixed-top', 'fixed-bottom', 'sticky-top'].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: [
                'navbar',
                {
                    [`bg-${props.color}`]: props.color,
                    [`navbar-${props.colorScheme}`]: props.colorScheme,
                    [typeof props.expand === 'boolean'
                        ? 'navbar-expand'
                        : `navbar-expand-${props.expand}`]: props.expand,
                },
                props.placement,
            ],
        }, props.container
            ? h$1('div', { class: [`container${props.container !== true ? '-' + props.container : ''}`] }, slots.default && slots.default())
            : slots.default && slots.default());
    },
});

const CNavbarBrand = defineComponent({
    name: 'CNavbarBrand',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         *
         */
        component: {
            type: String,
            default: 'a',
            required: false,
        },
        /**
         * The href attribute specifies the URL of the page the link goes to.
         */
        href: {
            type: String,
            default: undefined,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component ? props.component : props.href ? 'a' : 'span', {
            class: 'navbar-brand',
            href: props.href,
        }, slots.default && slots.default());
    },
});

const CNavbarNav = defineComponent({
    name: 'CNavbarNav',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'ul',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, {
            class: 'navbar-nav',
            role: 'navigation',
        }, slots.default && slots.default());
    },
});

const CNavbarText = defineComponent({
    name: 'CNavbarText',
    setup(_, { slots }) {
        return () => h$1('span', { class: 'navbar-text' }, slots.default && slots.default());
    },
});

const CNavbarToggler = defineComponent({
    name: 'CNavbarToggler',
    setup(_, { slots }) {
        return () => h$1('button', {
            class: 'navbar-toggler',
        }, slots.default ? slots.default() : h$1('span', { class: ['navbar-toggler-icon'] }));
    },
});

const CNavbarPlugin = {
    install: (app) => {
        app.component(CNavbar.name, CNavbar);
        app.component(CNavbarBrand.name, CNavbarBrand);
        app.component(CNavbarNav.name, CNavbarNav);
        app.component(CNavbarText.name, CNavbarText);
        app.component(CNavbarToggler.name, CNavbarToggler);
    },
};

const COffcanvas = defineComponent({
    name: 'COffcanvas',
    props: {
        /**
         * Apply a backdrop on body while offcanvas is open.
         *
         * @values boolean | 'static'
         */
        backdrop: {
            type: [Boolean, String],
            default: true,
            validator: (value) => {
                if (typeof value === 'string') {
                    return ['static'].includes(value);
                }
                if (typeof value === 'boolean') {
                    return true;
                }
                return false;
            },
        },
        /**
         * Closes the offcanvas when escape key is pressed.
         */
        keyboard: {
            type: Boolean,
            default: true,
        },
        /**
         * Components placement, there’s no default placement.
         *
         * @values 'start', 'end', 'top', 'bottom'
         */
        placement: {
            type: String,
            default: undefined,
            require: true,
            validator: (value) => {
                return ['start', 'end', 'top', 'bottom'].includes(value);
            },
        },
        /**
         * Responsive offcanvas property hide content outside the viewport from a specified breakpoint and down.
         *
         * @values boolean | 'sm' | 'md' | 'lg' | 'xl' | 'xxl'
         * @since 4.7.0
         */
        responsive: {
            type: [Boolean, String],
            default: true,
            validator: (value) => {
                if (typeof value === 'string') {
                    return ['sm', 'md', 'lg', 'xl', 'xxl'].includes(value);
                }
                if (typeof value === 'boolean') {
                    return true;
                }
                return false;
            },
        },
        /**
         * Allow body scrolling while offcanvas is open
         */
        scroll: {
            type: Boolean,
            default: false,
        },
        /**
         * Toggle the visibility of offcanvas component.
         */
        visible: Boolean,
    },
    emits: [
        /**
         * Callback fired when the component requests to be hidden.
         */
        'hide',
        /**
         * Callback fired when the component requests to be shown.
         */
        'show',
    ],
    setup(props, { slots, emit }) {
        const offcanvasRef = ref();
        const visible = ref(props.visible);
        watch(() => props.visible, () => {
            visible.value = props.visible;
        });
        watch(visible, () => {
            if (visible.value) {
                if (!props.scroll) {
                    document.body.style.overflow = 'hidden';
                    document.body.style.paddingRight = '0px';
                }
                return;
            }
            if (!props.scroll) {
                document.body.style.removeProperty('overflow');
                document.body.style.removeProperty('padding-right');
            }
        });
        const handleEnter = (el, done) => {
            emit('show');
            executeAfterTransition(() => done(), el);
            setTimeout(() => {
                el.classList.add('show');
            }, 1);
        };
        const handleAfterEnter = () => {
            offcanvasRef.value.focus();
        };
        const handleLeave = (el, done) => {
            executeAfterTransition(() => done(), el);
            el.classList.add('hiding');
        };
        const handleAfterLeave = (el) => {
            el.classList.remove('show', 'hiding');
        };
        const handleDismiss = () => {
            visible.value = false;
            emit('hide');
        };
        const handleBackdropDismiss = () => {
            if (props.backdrop !== 'static') {
                handleDismiss();
            }
        };
        const handleKeyDown = (event) => {
            if (event.key === 'Escape' && props.keyboard) {
                handleDismiss();
            }
        };
        return () => [
            h$1(Transition, {
                css: false,
                onEnter: (el, done) => handleEnter(el, done),
                onAfterEnter: () => handleAfterEnter(),
                onLeave: (el, done) => handleLeave(el, done),
                onAfterLeave: (el) => handleAfterLeave(el),
            }, () => withDirectives(h$1('div', {
                class: [
                    {
                        [`offcanvas${typeof props.responsive !== 'boolean' ? '-' + props.responsive : ''}`]: props.responsive,
                        [`offcanvas-${props.placement}`]: props.placement,
                    },
                ],
                onKeydown: (event) => handleKeyDown(event),
                ref: offcanvasRef,
                role: 'dialog',
                tabindex: -1,
            }, slots.default && slots.default()), [[vVisible, props.visible]])),
            props.backdrop &&
                h$1(CBackdrop, {
                    class: 'offcanvas-backdrop',
                    onClick: handleBackdropDismiss,
                    visible: visible.value,
                }),
        ];
    },
});

const COffcanvasBody = defineComponent({
    name: 'COffcanvasBody',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'offcanvas-body' }, slots.default && slots.default());
    },
});

const COffcanvasHeader = defineComponent({
    name: 'COffcanvasHeader',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'offcanvas-header' }, slots.default && slots.default());
    },
});

const COffcanvasTitle = defineComponent({
    name: 'COffcanvasTitle',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            required: false,
            default: 'h5',
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.component, { class: 'offcanvas-title' }, slots.default && slots.default());
    },
});

const COffcanvasPlugin = {
    install: (app) => {
        app.component(COffcanvas.name, COffcanvas);
        app.component(COffcanvasBody.name, COffcanvasBody);
        app.component(COffcanvasHeader.name, COffcanvasHeader);
        app.component(COffcanvasTitle.name, COffcanvasTitle);
    },
};

const CPagination = defineComponent({
    name: 'CPagination',
    props: {
        /**
         * Set the alignment of pagination components.
         *
         * @values 'start', 'center', 'end'
         */
        align: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['start', 'center', 'end'].includes(value);
            },
        },
        /**
         * Size the component small or large.
         *
         * @values 'sm', 'lg'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['sm', 'lg'].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1('nav', {}, h$1('ul', {
            class: [
                'pagination',
                {
                    [`justify-content-${props.align}`]: props.align,
                    [`pagination-${props.size}`]: props.size,
                },
            ],
        }, slots.default && slots.default()));
    },
});

const CPaginationItem = defineComponent({
    name: 'CPaginationItem',
    props: {
        /**
         * Toggle the active state for the component.
         */
        active: {
            type: Boolean,
            required: false,
        },
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Toggle the disabled state for the component.
         */
        disabled: {
            type: Boolean,
            required: false,
        },
        /**
         * The href attribute specifies the URL of the page the link goes to.
         */
        href: {
            type: String,
            default: undefined,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => {
            const component = props.component ? props.component : props.active ? 'span' : 'a';
            return h$1('li', {
                class: [
                    'page-item',
                    {
                        ['active']: props.active,
                        ['disabled']: props.disabled,
                    },
                ],
                ...(props.active && { active: props.active, 'aria-current': 'page' }),
            }, component === 'a'
                ? h$1(CLink, {
                    class: ['page-link'],
                    component: component,
                    href: props.href,
                }, {
                    default: () => slots.default && slots.default(),
                })
                : h$1(component, { class: ['page-link'] }, slots.default && slots.default()));
        };
    },
});

const CPaginationPlugin = {
    install: (app) => {
        app.component(CPagination.name, CPagination);
        app.component(CPaginationItem.name, CPaginationItem);
    },
};

const BREAKPOINTS$1 = [
    'xxl',
    'xl',
    'lg',
    'md',
    'sm',
    'xs',
];
const CPlaceholder = defineComponent({
    name: 'CPlaceholder',
    props: {
        /**
         * Set animation type to better convey the perception of something being actively loaded.
         *
         * @values 'glow', 'wave'
         */
        animation: {
            type: String,
            default: undefined,
            require: false,
            validator: (value) => {
                return ['glow', 'wave'].includes(value);
            },
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'span',
            required: false,
        },
        /**
         * Size the component extra small, small, or large.
         *
         * @values 'xs', 'sm', 'lg'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['xs', 'sm', 'lg'].includes(value);
            },
        },
        /**
         * The number of columns on extra small devices (<576px).
         */
        xs: {
            type: Number,
            default: undefined,
            require: false,
        },
        /**
         * The number of columns on small devices (<768px).
         */
        sm: {
            type: Number,
            default: undefined,
            require: false,
        },
        /**
         * The number of columns on medium devices (<992px).
         */
        md: {
            type: Number,
            default: undefined,
            require: false,
        },
        /**
         * The number of columns on large devices (<1200px).
         */
        lg: {
            type: Number,
            default: undefined,
            require: false,
        },
        /**
         * The number of columns on X-Large devices (<1400px).
         */
        xl: {
            type: Number,
            default: undefined,
            require: false,
        },
        /**
         * The number of columns on XX-Large devices (≥1400px).
         */
        xxl: {
            type: Number,
            default: undefined,
            require: false,
        },
    },
    setup(props, { slots }) {
        const repsonsiveClassNames = [];
        BREAKPOINTS$1.forEach((bp) => {
            const breakpoint = props[bp];
            const infix = bp === 'xs' ? '' : `-${bp}`;
            if (typeof breakpoint === 'number') {
                repsonsiveClassNames.push(`col${infix}-${breakpoint}`);
            }
            if (typeof breakpoint === 'boolean') {
                repsonsiveClassNames.push(`col${infix}`);
            }
        });
        return () => h$1(props.component, {
            class: [
                props.animation ? `placeholder-${props.animation}` : 'placeholder',
                {
                    [`bg-${props.color}`]: props.color,
                    [`placeholder-${props.size}`]: props.size,
                },
                repsonsiveClassNames,
            ],
        }, slots.default && slots.default());
    },
});

const CPlaceholderPlugin = {
    install: (app) => {
        app.component(CPlaceholder.name, CPlaceholder);
    },
};

const CProgressBar = defineComponent({
    name: 'CProgressBar',
    props: {
        /**
         * Use to animate the stripes right to left via CSS3 animations.
         */
        animated: {
            type: Boolean,
            required: false,
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * The percent to progress the ProgressBar.
         */
        value: {
            type: Number,
            required: false,
            default: 0,
        },
        /**
         * Set the progress bar variant to optional striped.
         *
         * @values 'striped'
         */
        variant: {
            type: String,
            default: undefined,
            require: false,
            validator: (value) => {
                return value === 'striped';
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1('div', {
            class: [
                'progress-bar',
                `bg-${props.color}`,
                {
                    [`progress-bar-${props.variant}`]: props.variant,
                    ['progress-bar-animated']: props.animated,
                },
            ],
            role: 'progressbar',
            style: `width: ${props.value}%`,
            'aria-valuenow': props.value,
            'aria-valuemin': '0',
            'aria-valuemax': '100',
        }, slots.default && slots.default());
    },
});

const CProgress = defineComponent({
    name: 'CProgress',
    props: {
        /**
         * Sets the height of the component. If you set that value the inner `<CProgressBar>` will automatically resize accordingly.
         */
        height: {
            type: Number,
            default: undefined,
            required: false,
        },
        /**
         * Makes progress bar thinner.
         */
        thin: Boolean,
        /**
         * Change the default color to white.
         */
        white: Boolean,
        ...CProgressBar.props,
    },
    setup(props, { slots }) {
        return () => h$1('div', {
            class: [
                'progress',
                {
                    'progress-thin': props.thin,
                    'progress-white': props.white,
                },
            ],
            ...(props.height, { style: `height: ${props.height}px` }),
        }, props.value
            ? h$1(CProgressBar, {
                value: props.value,
                animated: props.animated,
                color: props.color,
                variant: props.variant,
            }, slots.default && slots.default())
            : slots.default && slots.default());
    },
});

const CProgressPlugin = {
    install: (app) => {
        app.component(CProgress.name, CProgress);
        app.component(CProgressBar.name, CProgressBar);
    },
};

const CPopover = defineComponent({
    name: 'CPopover',
    props: {
        /**
         * Content for your component. If you want to pass non-string value please use dedicated slot `<template #content>...</template>`
         */
        content: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Offset of the popover relative to its target.
         */
        offset: {
            type: Array,
            default: () => [0, 8],
            required: false,
        },
        /**
         * Describes the placement of your component after Popper.js has applied all the modifiers that may have flipped or altered the originally provided placement property.
         */
        placement: {
            type: String,
            default: 'top',
            required: false,
            validator: (value) => {
                return ['top', 'right', 'bottom', 'left'].includes(value);
            },
        },
        /**
         * Title for your component. If you want to pass non-string value please use dedicated slot `<template #title>...</template>`
         */
        title: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Sets which event handlers you’d like provided to your toggle prop. You can specify one trigger or an array of them.
         *
         * @values 'click', 'focus', 'hover'
         */
        trigger: {
            type: [String, Array],
            default: 'click',
            required: false,
            validator: (value) => {
                if (typeof value === 'string') {
                    return ['click', 'focus', 'hover'].includes(value);
                }
                if (Array.isArray(value)) {
                    return value.every((e) => ['click', 'focus', 'hover'].includes(e));
                }
                return false;
            },
        },
        /**
         * Toggle the visibility of popover component.
         */
        visible: Boolean,
    },
    emits: [
        /**
         * Callback fired when the component requests to be hidden.
         */
        'hide',
        /**
         * Callback fired when the component requests to be shown.
         */
        'show',
    ],
    setup(props, { slots, emit }) {
        const togglerRef = ref();
        const popoverRef = ref();
        const popper = ref();
        const visible = ref(props.visible);
        const handleEnter = (el, done) => {
            emit('show');
            initPopper();
            el.classList.add('show');
            executeAfterTransition(() => done(), el);
        };
        const handleLeave = (el, done) => {
            emit('hide');
            el.classList.remove('show');
            executeAfterTransition(() => {
                done();
                destroyPopper();
            }, el);
        };
        const handleToggle = (event) => {
            togglerRef.value = event.target;
            visible.value = !visible.value;
        };
        const initPopper = () => {
            if (togglerRef.value) {
                popper.value = createPopper(togglerRef.value, popoverRef.value, {
                    placement: props.placement,
                    modifiers: [
                        {
                            name: 'offset',
                            options: {
                                offset: props.offset,
                            },
                        },
                    ],
                });
            }
        };
        const destroyPopper = () => {
            if (popper.value) {
                popper.value.destroy();
            }
            popper.value = undefined;
        };
        return () => [
            h$1(Teleport, {
                to: 'body',
            }, h$1(Transition, {
                onEnter: (el, done) => handleEnter(el, done),
                onLeave: (el, done) => handleLeave(el, done),
            }, () => visible.value &&
                h$1('div', {
                    class: 'popover fade bs-popover-auto',
                    ref: popoverRef,
                    role: 'tooltip',
                }, [
                    h$1('div', { class: 'popover-arrow', 'data-popper-arrow': '' }),
                    (props.title || slots.title) &&
                        h$1('div', { class: 'popover-header' }, {
                            default: () => (slots.title && slots.title()) || props.title,
                        }),
                    (props.content || slots.content) &&
                        h$1('div', { class: 'popover-body' }, {
                            default: () => (slots.content && slots.content()) || props.content,
                        }),
                ]))),
            slots.toggler &&
                slots.toggler({
                    on: {
                        click: (event) => props.trigger.includes('click') && handleToggle(event),
                        blur: (event) => props.trigger.includes('focus') && handleToggle(event),
                        focus: (event) => props.trigger.includes('focus') && handleToggle(event),
                        mouseenter: (event) => props.trigger.includes('hover') && handleToggle(event),
                        mouseleave: (event) => props.trigger.includes('hover') && handleToggle(event),
                    },
                }),
        ];
    },
});

const CPopoverPlugin = {
    install: (app) => {
        app.component(CPopover.name, CPopover);
    },
};

const isOnMobile = (element) => Boolean(getComputedStyle(element).getPropertyValue('--cui-is-mobile'));
const CSidebar = defineComponent({
    name: 'CSidebar',
    props: {
        /**
         * Make sidebar narrow.
         */
        narrow: {
            type: Boolean,
            required: false,
        },
        /**
         * Set sidebar to overlaid variant.
         */
        overlaid: {
            type: Boolean,
            required: false,
        },
        /**
         * Place sidebar in non-static positions.
         */
        position: {
            type: String,
            default: undefined,
            validator: (value) => {
                return ['fixed'].includes(value);
            },
        },
        /**
         * Size the component small, large, or extra large.
         */
        size: {
            type: String,
            default: undefined,
            validator: (value) => {
                return ['sm', 'lg', 'xl'].includes(value);
            },
        },
        /**
         * Expand narrowed sidebar on hover.
         */
        unfoldable: Boolean,
        /**
         * Toggle the visibility of sidebar component.
         */
        visible: Boolean,
    },
    emits: [
        /**
         * Callback fired when the component requests to be hidden.
         */
        'hide',
        /**
         * Callback fired when the component requests to be shown.
         */
        'show',
        /**
         * Event emitted after visibility of component changed.
         */
        'visible-change',
    ],
    setup(props, { attrs, slots, emit }) {
        const mobile = ref();
        const inViewport = ref();
        const sidebarRef = ref();
        const visible = ref(props.visible);
        watch(inViewport, () => {
            emit('visible-change', inViewport.value);
            inViewport.value ? emit('show') : emit('hide');
        });
        watch(() => props.visible, () => (visible.value = props.visible));
        watch(mobile, () => {
            if (mobile.value && visible.value)
                visible.value = false;
        });
        onMounted(() => {
            mobile.value = isOnMobile(sidebarRef.value);
            inViewport.value = isInViewport(sidebarRef.value);
            window.addEventListener('resize', () => handleResize());
            window.addEventListener('mouseup', handleClickOutside);
            window.addEventListener('keyup', handleKeyup);
            sidebarRef.value.addEventListener('mouseup', handleOnClick);
            sidebarRef.value.addEventListener('transitionend', () => {
                inViewport.value = isInViewport(sidebarRef.value);
            });
        });
        onBeforeUnmount(() => {
            window.removeEventListener('resize', () => handleResize());
            window.removeEventListener('mouseup', handleClickOutside);
            window.removeEventListener('keyup', handleKeyup);
            sidebarRef.value.removeEventListener('mouseup', handleOnClick);
            sidebarRef.value.removeEventListener('transitionend', () => {
                inViewport.value = isInViewport(sidebarRef.value);
            });
        });
        const handleHide = () => {
            visible.value = false;
            emit('visible-change', false);
        };
        const handleResize = () => {
            mobile.value = isOnMobile(sidebarRef.value);
            inViewport.value = isInViewport(sidebarRef.value);
        };
        const handleKeyup = (event) => {
            if (mobile.value && !sidebarRef.value.contains(event.target)) {
                handleHide();
            }
        };
        const handleClickOutside = (event) => {
            if (mobile.value && !sidebarRef.value.contains(event.target)) {
                handleHide();
            }
        };
        const handleOnClick = (event) => {
            const target = event.target;
            target &&
                target.classList.contains('nav-link') &&
                !target.classList.contains('nav-group-toggle') &&
                mobile.value &&
                handleHide();
        };
        return () => [
            h$1('div', {
                class: [
                    'sidebar',
                    {
                        'sidebar-narrow': props.narrow,
                        'sidebar-overlaid': props.overlaid,
                        [`sidebar-${props.position}`]: props.position,
                        [`sidebar-${props.size}`]: props.size,
                        'sidebar-narrow-unfoldable': props.unfoldable,
                        show: visible.value === true && mobile.value,
                        hide: visible.value === false && !mobile.value,
                    },
                    attrs.class,
                ],
                ref: sidebarRef,
            }, slots.default && slots.default()),
            mobile.value &&
                h$1(CBackdrop, {
                    class: 'sidebar-backdrop d-none',
                    visible: props.visible,
                    onClick: () => handleHide(),
                }),
        ];
    },
});

const CSidebarBrand = defineComponent({
    name: 'CSidebarBrand',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'sidebar-brand' }, slots.default && slots.default());
    },
});

const CSidebarFooter = defineComponent({
    name: 'CSidebarFooter',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'sidebar-footer' }, slots.default && slots.default());
    },
});

const CSidebarHeader = defineComponent({
    name: 'CSidebarHeader',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'sidebar-header' }, slots.default && slots.default());
    },
});

const CSidebarNav = defineComponent({
    name: 'CSidebarNav',
    setup(_, { slots }) {
        const visibleGroup = ref();
        const handleVisibleChange = (visible, index) => {
            if (visible) {
                visibleGroup.value = index;
            }
            else {
                if (visibleGroup.value === index) {
                    visibleGroup.value = 0;
                }
            }
        };
        const isVisible = (index) => Boolean(visibleGroup.value === index);
        return () => h$1('ul', {
            class: 'sidebar-nav',
        }, slots.default &&
            slots.default().map((vnode, index) => {
                // @ts-expect-error name is defined in component
                if (vnode.type.name === 'CNavGroup') {
                    return h$1(vnode, {
                        onVisibleChange: (visible) => handleVisibleChange(visible, index + 1),
                        ...(visibleGroup.value && { visible: isVisible(index + 1) }),
                    });
                }
                return vnode;
            }));
    },
});

const CSidebarToggler = defineComponent({
    name: 'CSidebarToggler',
    setup(_, { slots }) {
        return () => h$1('button', { class: 'sidebar-toggler' }, slots.default && slots.default());
    },
});

const CSidebarPlugin = {
    install: (app) => {
        app.component(CSidebar.name, CSidebar);
        app.component(CSidebarBrand.name, CSidebarBrand);
        app.component(CSidebarFooter.name, CSidebarFooter);
        app.component(CSidebarHeader.name, CSidebarHeader);
        app.component(CSidebarNav.name, CSidebarNav);
        app.component(CSidebarToggler.name, CSidebarToggler);
    },
};

const CSpinner = defineComponent({
    name: 'CSpinner',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return [
                    'primary',
                    'secondary',
                    'success',
                    'danger',
                    'warning',
                    'info',
                    'dark',
                    'light',
                ].includes(value);
            },
        },
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: 'div',
            required: false,
        },
        /**
         * Size the component small.
         *
         * @values 'sm'
         */
        size: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return value === 'sm';
            },
        },
        /**
         * Set the button variant to an outlined button or a ghost button.
         *
         * @values 'border', 'grow'
         */
        variant: {
            type: String,
            default: 'border',
            required: false,
            validator: (value) => {
                return ['border', 'grow'].includes(value);
            },
        },
        /**
         * Set visually hidden label for accessibility purposes.
         */
        visuallyHiddenLabel: {
            type: String,
            default: 'Loading...',
            required: false,
        },
    },
    setup(props) {
        return () => h$1(props.component, {
            class: [
                `spinner-${props.variant}`,
                `text-${props.color}`,
                props.size && `spinner-${props.variant}-${props.size}`,
            ],
            role: 'status',
        }, h$1('span', { class: ['visually-hidden'] }, props.visuallyHiddenLabel));
    },
});

const CSpinnerPlugin = {
    install: (app) => {
        app.component(CSpinner.name, CSpinner);
    },
};

const CTableBody = defineComponent({
    name: 'CTableBody',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values  'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
    },
    setup(props, { slots }) {
        return () => h$1('tbody', {
            class: [
                {
                    [`table-${props.color}`]: props.color,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CTableCaption = defineComponent({
    name: 'CTableCaption',
    setup(_, { slots }) {
        return () => h$1('caption', {}, slots.default && slots.default());
    },
});

const CTableDataCell = defineComponent({
    name: 'CTableDataCell',
    props: {
        /**
         * Highlight a table row or cell.
         */
        active: {
            type: Boolean,
            required: false,
        },
        /**
         * Set the vertical aligment.
         *
         * @values 'bottom', 'middle', 'top'
         */
        align: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['bottom', 'middle', 'top'].includes(value);
            },
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values  'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
        /**
         * @ignore
         */
        scope: {
            type: String,
            required: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(props.scope ? 'th' : 'td', {
            class: [
                {
                    [`align-${props.align}`]: props.align,
                    'table-active': props.active,
                    [`table-${props.color}`]: props.color,
                },
            ],
            ...(props.scope && { scope: props.scope }),
        }, slots.default && slots.default());
    },
});

const CTableFoot = defineComponent({
    name: 'CTableFoot',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values  'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
    },
    setup(props, { slots }) {
        return () => h$1('tfoot', {
            class: [
                {
                    [`table-${props.color}`]: props.color,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CTableHead = defineComponent({
    name: 'CTableHead',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values  'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
    },
    setup(props, { slots }) {
        return () => h$1('thead', {
            class: [
                {
                    [`table-${props.color}`]: props.color,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CTableHeaderCell = defineComponent({
    name: 'CTableHeaderCell',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values  'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
    },
    setup(props, { slots }) {
        return () => h$1('th', {
            class: [
                {
                    [`table-${props.color}`]: props.color,
                },
            ],
        }, slots.default && slots.default());
    },
});

const CTableRow = defineComponent({
    name: 'CTableRow',
    props: {
        /**
         * Highlight a table row or cell..
         */
        active: {
            type: Boolean,
            required: false,
        },
        /**
         * Set the vertical aligment.
         *
         * @values 'bottom', 'middle', 'top'
         */
        align: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['bottom', 'middle', 'top'].includes(value);
            },
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values  'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
    },
    setup(props, { slots }) {
        return () => h$1('tr', {
            class: [
                {
                    [`align-${props.align}`]: props.align,
                    'table-active': props.active,
                    [`table-${props.color}`]: props.color,
                },
            ],
        }, slots.default && slots.default());
    },
});

const pretifyName = (name) => {
    return name
        .replace(/[-_.]/g, ' ')
        .replace(/ +/g, ' ')
        .replace(/([a-z0-9])([A-Z])/g, '$1 $2')
        .split(' ')
        .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
        .join(' ');
};
const label = (column) => typeof column === 'object'
    ? column.label !== undefined
        ? column.label
        : pretifyName(column.key)
    : pretifyName(column);
const CTable = defineComponent({
    name: 'CTable',
    props: {
        /**
         * Set the vertical aligment.
         *
         * @values 'bottom', 'middle', 'top'
         */
        align: {
            type: String,
            default: undefined,
            required: false,
            validator: (value) => {
                return ['bottom', 'middle', 'top'].includes(value);
            },
        },
        /**
         * Sets the border color of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        borderColor: Color,
        /**
         * Add borders on all sides of the table and cells.
         */
        bordered: {
            type: Boolean,
            required: false,
        },
        /**
         * Remove borders on all sides of the table and cells.
         */
        borderless: {
            type: Boolean,
            required: false,
        },
        /**
         * Put the `<caption>` on the top of the table.
         *
         * @values 'top' | string
         */
        caption: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Set the text of the table caption and the caption on the top of the table.
         *
         * @since 4.5.0
         */
        captionTop: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Prop for table columns configuration. If prop is not defined, table will display columns based on the first item keys, omitting keys that begins with underscore (e.g. '_props')
         *
         * In columns prop each array item represents one column. Item might be specified in two ways:
         * String: each item define column name equal to item value.
         * Object: item is object with following keys available as column configuration:
         * - key (required)(String) - define column name equal to item key.
         * - label (String) - define visible label of column. If not defined, label will be generated automatically based on column name, by converting kebab-case and snake_case to individual words and capitalization of each word.
         * - _props (Object) - adds classes to all cels in column, ex. _props: { scope: 'col', className: 'custom-class' },
         * - _style (Object) - adds styles to the column header (useful for defining widths)
         *
         * @since 4.5.0
         */
        columns: {
            type: Array,
            required: false,
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values  'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
        /**
         * Array of objects or strings, where each element represents one cell in the table footer.
         *
         * Example items:
         * ['FooterCell', 'FooterCell', 'FooterCell']
         * or
         * [{ label: 'FooterCell', _props: { color: 'success' }, ...]
         *
         * @since 4.5.0
         */
        footer: {
            type: Array,
            required: false,
        },
        /**
         * Enable a hover state on table rows within a `<CTableBody>`.
         */
        hover: {
            type: Boolean,
            required: false,
        },
        /**
         * Array of objects, where each object represents one item - row in table. Additionally, you can add style classes to each row by passing them by '_props' key and to single cell by '_cellProps'.
         *
         * Example item:
         * { name: 'John' , age: 12, _props: { color: 'success' }, _cellProps: { age: { className: 'fw-bold'}}}
         *
         * @since 4.5.0
         */
        items: {
            type: Array,
            required: false,
        },
        responsive: {
            type: [Boolean, String],
            default: undefined,
            required: false,
            validator: (value) => {
                if (typeof value == 'string') {
                    return ['sm', 'md', 'lg', 'xl', 'xxl'].includes(value);
                }
                if (typeof value == 'boolean') {
                    return true;
                }
                return false;
            },
        },
        /**
         * Make table more compact by cutting all cell `padding` in half.
         */
        small: {
            type: Boolean,
            required: false,
        },
        /**
         * Add zebra-striping to any table row within the `<CTableBody>`.
         */
        striped: {
            type: Boolean,
            required: false,
        },
        /**
         * Add zebra-striping to any table column.
         *
         * @since 4.4.0
         */
        stripedColumns: {
            type: Boolean,
            required: false,
        },
        /**
         * Properties that will be passed to the table footer component.
         *
         * Properties to [CTableFoot](#ctablefoot) component.
         * @since 4.5.0
         */
        tableFootProps: {
            type: Object,
            default: undefined,
            required: false,
        },
        /**
         * Properties that will be passed to the table head component.
         *
         *  Properties to [CTableHead](#ctablehead) component.
         * @since 4.5.0
         */
        tableHeadProps: {
            type: Object,
            default: undefined,
            required: false,
        },
    },
    setup(props, { slots, attrs }) {
        const rawColumnNames = computed(() => props.columns
            ? props.columns.map((column) => {
                if (typeof column === 'object')
                    return column.key;
                else
                    return column;
            })
            : Object.keys((props.items && props.items[0]) || {}).filter((el) => el.charAt(0) !== '_'));
        const table = () => h$1('table', {
            class: [
                'table',
                {
                    [`align-${props.align}`]: props.align,
                    [`caption-top`]: props.captionTop || props.caption === 'top',
                    [`border-${props.borderColor}`]: props.borderColor,
                    'table-bordered': props.bordered,
                    'table-borderless': props.borderless,
                    [`table-${props.color}`]: props.color,
                    'table-hover': props.hover,
                    'table-sm': props.small,
                    'table-striped': props.striped,
                    'table-striped-columns': props.stripedColumns,
                },
                attrs.class,
            ],
        }, {
            default: () => [
                ((props.caption && props.caption !== 'top') || props.captionTop) &&
                    h$1(CTableCaption, {}, {
                        default: () => props.caption || props.captionTop,
                    }),
                props.columns &&
                    h$1(CTableHead, {
                        ...props.tableHeadProps,
                    }, {
                        default: () => h$1(CTableRow, {}, {
                            default: () => [
                                props.columns &&
                                    props.columns.map((column) => h$1(CTableHeaderCell, {
                                        ...(typeof column === 'object' &&
                                            column._props && { ...column._props }),
                                        ...(typeof column === 'object' &&
                                            column._style && { style: { ...column._style } }),
                                    }, {
                                        default: () => label(column),
                                    })),
                            ],
                        }),
                    }),
                props.items &&
                    h$1(CTableBody, {}, {
                        default: () => [
                            props.items &&
                                props.items.map((item) => h$1(CTableRow, {
                                    ...(item._props && { ...item._props }),
                                }, {
                                    default: () => [
                                        rawColumnNames.value.map((colName) => item[colName] &&
                                            h$1(CTableDataCell, {
                                                ...(item._cellProps &&
                                                    item._cellProps['all'] && { ...item._cellProps['all'] }),
                                                ...(item._cellProps &&
                                                    item._cellProps[colName] && {
                                                    ...item._cellProps[colName],
                                                }),
                                            }, {
                                                default: () => item[colName],
                                            })),
                                    ],
                                })),
                        ],
                    }),
                slots.default && slots.default(),
                props.footer &&
                    h$1(CTableFoot, {
                        ...props.tableFootProps,
                    }, {
                        default: () => h$1(CTableRow, {}, {
                            default: () => [
                                props.footer &&
                                    props.footer.map((item) => h$1(CTableDataCell, {
                                        ...(typeof item === 'object' &&
                                            item._props && { ...item._props }),
                                    }, {
                                        default: () => (typeof item === 'object' ? item.label : item),
                                    })),
                            ],
                        }),
                    }),
            ],
        });
        return () => [
            props.responsive
                ? h$1('div', {
                    class: typeof props.responsive === 'boolean'
                        ? 'table-responsive'
                        : `table-responsive-${props.responsive}`,
                }, table())
                : table(),
        ];
    },
});

const CTablePlugin = {
    install: (app) => {
        app.component(CTable.name, CTable);
        app.component(CTableBody.name, CTableBody);
        app.component(CTableCaption.name, CTableCaption);
        app.component(CTableDataCell.name, CTableDataCell);
        app.component(CTableFoot.name, CTableFoot);
        app.component(CTableHead.name, CTableHead);
        app.component(CTableHeaderCell.name, CTableHeaderCell);
        app.component(CTableRow.name, CTableRow);
    },
};

const CTabContent = defineComponent({
    name: 'CTabContent',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'tab-content' }, slots.default && slots.default());
    },
});

const CTabPane = defineComponent({
    name: 'CTabPane',
    props: {
        /**
         * Toggle the visibility of component.
         */
        visible: {
            type: Boolean,
            default: false,
            required: false,
        },
    },
    emits: [
        /**
         * Callback fired when the component requests to be hidden.
         */
        'hide',
        /**
         * Callback fired when the component requests to be shown.
         */
        'show',
    ],
    setup(props, { slots, emit }) {
        const tabPaneRef = ref();
        const firstRender = ref(true);
        const handleEnter = (el, done) => {
            firstRender.value = false;
            emit('show');
            setTimeout(() => {
                executeAfterTransition(() => done(), el);
                el.classList.add('show');
            }, 1);
        };
        const handleLeave = (el, done) => {
            firstRender.value = false;
            emit('hide');
            el.classList.remove('show');
            executeAfterTransition(() => done(), el);
        };
        return () => h$1(Transition, {
            onEnter: (el, done) => handleEnter(el, done),
            onLeave: (el, done) => handleLeave(el, done),
        }, () => withDirectives(h$1('div', {
            class: [
                'tab-pane',
                'fade',
                {
                    active: props.visible,
                    show: firstRender.value && props.visible,
                },
            ],
            ref: tabPaneRef,
        }, slots.default && slots.default()), [[vShow, props.visible]]));
    },
});

const CTabsPlugin = {
    install: (app) => {
        app.component(CTabContent.name, CTabContent);
        app.component(CTabPane.name, CTabPane);
    },
};

const CToast = defineComponent({
    name: 'CToast',
    props: {
        /**
         * Auto hide the toast.
         */
        autohide: {
            type: Boolean,
            default: true,
            required: false,
        },
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light', string
         */
        color: Color,
        /**
         * Delay hiding the toast (ms).
         */
        delay: {
            type: Number,
            default: 5000,
            required: false,
        },
        /**
         * Optionally add a close button to component and allow it to self dismiss.
         */
        dismissible: {
            type: Boolean,
            default: true,
            required: false,
        },
        /**
         * index of the component.
         */
        index: {
            type: Number,
            default: undefined,
            required: false,
        },
        /**
         * Title node for your component.
         */
        title: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Toggle the visibility of component.
         */
        visible: {
            type: Boolean,
            default: true,
            required: false,
        },
    },
    emits: [
        /**
         * Callback fired when the component requests to be closed.
         */
        'close',
        /**
         * Callback fired when the component requests to be shown.
         */
        'show',
    ],
    setup(props, { slots, emit }) {
        const visible = ref(props.visible);
        let timeout = 0;
        const updateVisible = (v) => {
            visible.value = v;
        };
        provide('updateVisible', updateVisible);
        const handleBeforeEnter = (el) => {
            el.classList.add('showing');
        };
        const handleEnter = (el, done) => {
            executeAfterTransition(() => done(), el);
            el.classList.add('show');
            setTimeout(() => {
                el.classList.remove('showing');
            }, 1);
            if (props.index) {
                emit('show', props.index);
            }
            else {
                emit('show');
            }
        };
        const handleLeave = (el, done) => {
            executeAfterTransition(() => done(), el);
            el.classList.add('showing');
        };
        const handleAfterLeave = (el) => {
            el.classList.remove('show');
            el.classList.add('hide');
            if (props.index) {
                emit('close', props.index);
            }
            else {
                emit('close');
            }
        };
        onMounted(() => {
            if (props.autohide) {
                clearTimeout(timeout);
                timeout = window.setTimeout(() => {
                    visible.value = false;
                    emit('close');
                }, props.delay);
            }
        });
        return () => h$1(Transition, {
            appear: true,
            onBeforeEnter: (el) => handleBeforeEnter(el),
            onEnter: (el, done) => handleEnter(el, done),
            onLeave: (el, done) => handleLeave(el, done),
            onAfterLeave: (el) => handleAfterLeave(el),
        }, () => visible.value &&
            h$1('div', {
                class: [
                    'toast fade',
                    {
                        [`bg-${props.color}`]: props.color,
                    },
                ],
                'aria-live': 'assertive',
                'aria-atomic': true,
                role: 'alert',
                ref: 'toastRef',
            }, slots.default && slots.default()));
    },
});

const CToastBody = defineComponent({
    name: 'CToastBody',
    setup(_, { slots }) {
        return () => h$1('div', { class: 'toast-body' }, slots.default && slots.default());
    },
});

const CToastClose = defineComponent({
    name: 'CToastClose',
    props: {
        /**
         * Component used for the root node. Either a string to use a HTML element or a component.
         */
        component: {
            type: String,
            default: undefined,
            required: false,
        },
        ...CCloseButton.props,
    },
    emits: [
        /**
         * Event called before the dissmiss animation has started.
         */
        'close',
    ],
    setup(props, { slots, emit }) {
        // eslint-disable-next-line no-unused-vars
        const updateVisible = inject('updateVisible');
        const handleClose = () => {
            emit('close');
            updateVisible(false);
        };
        return () => props.component
            ? h$1(props.component, {
                onClick: () => {
                    handleClose();
                },
            }, () => slots.default && slots.default())
            : h$1(CCloseButton, {
                ...props,
                onClick: () => {
                    handleClose();
                },
            });
    },
});

const CToaster = defineComponent({
    name: 'CToaster',
    props: {
        /**
         * Describes the placement of component.
         *
         * @values 'top-start', 'top', 'top-end', 'middle-start', 'middle', 'middle-end', 'bottom-start', 'bottom', 'bottom-end'
         */
        placement: {
            type: String,
            default: undefined,
            require: false,
            validator: (value) => {
                return [
                    'top-start',
                    'top-center',
                    'top-end',
                    'middle-start',
                    'middle-center',
                    'middle-end',
                    'bottom-start',
                    'bottom-center',
                    'bottom-end',
                ].includes(value);
            },
        },
    },
    setup(props, { slots }) {
        return () => h$1('div', {
            class: [
                'toaster toast-container p-3',
                {
                    'position-fixed': props.placement,
                    'top-0': props.placement && props.placement.includes('top'),
                    'top-50 translate-middle-y': props.placement && props.placement.includes('middle'),
                    'bottom-0': props.placement && props.placement.includes('bottom'),
                    'start-0': props.placement && props.placement.includes('start'),
                    'start-50 translate-middle-x': props.placement && props.placement.includes('center'),
                    'end-0': props.placement && props.placement.includes('end'),
                },
            ],
        }, slots.default && slots.default());
    },
});

const CToastHeader = defineComponent({
    name: 'CToastHeader',
    props: {
        /**
         * Automatically add a close button to the header.
         */
        closeButton: {
            type: Boolean,
            require: false,
        },
    },
    emits: [
        /**
         * Event called after clicking the close button.
         */
        'close',
    ],
    setup(props, { slots, emit }) {
        return () => h$1('div', { class: 'toast-header' }, [
            slots.default && slots.default(),
            props.closeButton &&
                h$1(CToastClose, {
                    onClose: () => emit('close'),
                }),
        ]);
    },
});

const CToastPlugin = {
    install: (app) => {
        app.component(CToast.name, CToast);
        app.component(CToastBody.name, CToastBody);
        app.component(CToastClose.name, CToastClose);
        app.component(CToaster.name, CToaster);
        app.component(CToastHeader.name, CToastHeader);
    },
};

const CTooltip = defineComponent({
    name: 'CTooltip',
    props: {
        /**
         * Content for your component. If you want to pass non-string value please use dedicated slot `<template #content>...</template>`
         */
        content: {
            type: String,
            default: undefined,
            required: false,
        },
        /**
         * Offset of the tooltip relative to its target.
         */
        offset: {
            type: Array,
            default: () => [0, 0],
            required: false,
        },
        /**
         * Describes the placement of your component after Popper.js has applied all the modifiers that may have flipped or altered the originally provided placement property.
         */
        placement: {
            type: String,
            default: 'top',
            required: false,
            validator: (value) => {
                return ['top', 'right', 'bottom', 'left'].includes(value);
            },
        },
        /**
         * Sets which event handlers you’d like provided to your toggle prop. You can specify one trigger or an array of them.
         *
         * @values 'click', 'focus', 'hover'
         */
        trigger: {
            type: [String, Array],
            default: 'hover',
            required: false,
            validator: (value) => {
                if (typeof value === 'string') {
                    return ['click', 'focus', 'hover'].includes(value);
                }
                if (Array.isArray(value)) {
                    return value.every((e) => ['click', 'focus', 'hover'].includes(e));
                }
                return false;
            },
        },
        /**
         * Toggle the visibility of tooltip component.
         */
        visible: Boolean,
    },
    emits: [
        /**
         * Callback fired when the component requests to be hidden.
         */
        'hide',
        /**
         * Callback fired when the component requests to be shown.
         */
        'show',
    ],
    setup(props, { slots, emit }) {
        const togglerRef = ref();
        const tooltipRef = ref();
        const popper = ref();
        const visible = ref(props.visible);
        const handleEnter = (el, done) => {
            emit('show');
            initPopper();
            el.classList.add('show');
            executeAfterTransition(() => done(), el);
        };
        const handleLeave = (el, done) => {
            emit('hide');
            el.classList.remove('show');
            executeAfterTransition(() => {
                done();
                destroyPopper();
            }, el);
        };
        const handleToggle = (event) => {
            togglerRef.value = event.target;
            visible.value = !visible.value;
        };
        const initPopper = () => {
            if (togglerRef.value) {
                popper.value = createPopper(togglerRef.value, tooltipRef.value, {
                    placement: props.placement,
                    modifiers: [
                        {
                            name: 'offset',
                            options: {
                                offset: props.offset,
                            },
                        },
                    ],
                });
            }
        };
        const destroyPopper = () => {
            if (popper.value) {
                popper.value.destroy();
            }
            popper.value = undefined;
        };
        return () => [
            h$1(Teleport, {
                to: 'body',
            }, h$1(Transition, {
                onEnter: (el, done) => handleEnter(el, done),
                onLeave: (el, done) => handleLeave(el, done),
            }, () => visible.value &&
                h$1('div', {
                    class: 'tooltip fade bs-tooltip-auto',
                    ref: tooltipRef,
                    role: 'tooltip',
                }, [
                    h$1('div', { class: 'tooltip-arrow', 'data-popper-arrow': '' }),
                    (props.content || slots.content) &&
                        h$1('div', { class: 'tooltip-inner' }, {
                            default: () => (slots.content && slots.content()) || props.content,
                        }),
                ]))),
            slots.toggler &&
                slots.toggler({
                    on: {
                        click: (event) => props.trigger.includes('click') && handleToggle(event),
                        blur: (event) => props.trigger.includes('focus') && handleToggle(event),
                        focus: (event) => props.trigger.includes('focus') && handleToggle(event),
                        mouseenter: (event) => props.trigger.includes('hover') && handleToggle(event),
                        mouseleave: (event) => props.trigger.includes('hover') && handleToggle(event),
                    },
                }),
        ];
    },
});

const CTooltipPlugin = {
    install: (app) => {
        app.component(CTooltip.name, CTooltip);
    },
};

const CWidgetStatsA = defineComponent({
    name: 'CWidgetStatsA',
    props: {
        color: {
            type: String,
            default: undefined,
            require: false,
        },
        /**
         * Title for your component. If you want to pass non-string value please use dedicated slot `<template #title>...</template>`
         */
        title: {
            type: String,
            default: undefined,
            require: false,
        },
        /**
         * Value for your component. If you want to pass non-string or non-number value please use dedicated slot `<template #value>...</template>`
         */
        value: {
            type: [Number, String],
            default: 0,
            require: false,
        },
    },
    /**
     * Location for action component, ex. `<CDropdown>`.
     *
     * @slot action
     */
    /**
     * Location for chart component.
     *
     * @slot chart
     */
    setup(props, { slots }) {
        return () => h$1(CCard, {
            class: [
                { [`bg-${props.color}`]: props.color, 'text-high-emphasis-inverse': props.color },
            ],
        }, () => [
            h$1(CCardBody, {
                class: 'pb-0 d-flex justify-content-between align-items-start',
            }, () => [
                h$1('div', {}, [
                    (props.value || slots.value) &&
                        h$1('div', { class: 'fs-4 fw-semibold' }, {
                            default: () => (slots.value && slots.value()) || props.value,
                        }),
                    (props.title || slots.title) &&
                        h$1('div', {}, {
                            default: () => (slots.title && slots.title()) || props.title,
                        }),
                ]),
                /**
                 * @slot Location for action component, ex. `<CDropdown>`.
                 */
                slots.action && slots.action(),
            ]),
            /**
             * @slot Location for chart component.
             */
            slots.chart && slots.chart(),
            slots.default && slots.default(),
        ]);
    },
});

/*!
 * is-plain-object <https://github.com/jonschlinkert/is-plain-object>
 *
 * Copyright (c) 2014-2017, Jon Schlinkert.
 * Released under the MIT License.
 */

function isObject(o) {
  return Object.prototype.toString.call(o) === '[object Object]';
}

function isPlainObject(o) {
  var ctor,prot;

  if (isObject(o) === false) return false;

  // If has modified constructor
  ctor = o.constructor;
  if (ctor === undefined) return true;

  // If has modified prototype
  prot = ctor.prototype;
  if (isObject(prot) === false) return false;

  // If constructor does not have an Object-specific method
  if (prot.hasOwnProperty('isPrototypeOf') === false) {
    return false;
  }

  // Most likely a plain Object
  return true;
}

function t(){return t=Object.assign?Object.assign.bind():function(e){for(var t=1;t<arguments.length;t++){var n=arguments[t];for(var r in n)Object.prototype.hasOwnProperty.call(n,r)&&(e[r]=n[r]);}return e},t.apply(this,arguments)}function n(e,t){if(null==e)return {};var n,r,i={},o=Object.keys(e);for(r=0;r<o.length;r++)t.indexOf(n=o[r])>=0||(i[n]=e[n]);return i}const r={silent:!1,logLevel:"warn"},i=["validator"],o=Object.prototype,a=o.toString,s=o.hasOwnProperty,u=/^\s*function (\w+)/;function l(e){var t;const n=null!==(t=null==e?void 0:e.type)&&void 0!==t?t:e;if(n){const e=n.toString().match(u);return e?e[1]:""}return ""}const c=isPlainObject,d=e=>e;let f=d;if("production"!==process.env.NODE_ENV){const e="undefined"!=typeof console;f=e?function(e,t=r.logLevel){!1===r.silent&&console[t](`[VueTypes warn]: ${e}`);}:d;}const p=(e,t)=>s.call(e,t),y=Number.isInteger||function(e){return "number"==typeof e&&isFinite(e)&&Math.floor(e)===e},v=Array.isArray||function(e){return "[object Array]"===a.call(e)},h=e=>"[object Function]"===a.call(e),b=(e,t)=>c(e)&&p(e,"_vueTypes_name")&&(!t||e._vueTypes_name===t),g=e=>c(e)&&(p(e,"type")||["_vueTypes_name","validator","default","required"].some(t=>p(e,t)));function O(e,t){return Object.defineProperty(e.bind(t),"__original",{value:e})}function m(e,t,n=!1){let r,i=!0,o="";r=c(e)?e:{type:e};const a=b(r)?r._vueTypes_name+" - ":"";if(g(r)&&null!==r.type){if(void 0===r.type||!0===r.type)return i;if(!r.required&&void 0===t)return i;v(r.type)?(i=r.type.some(e=>!0===m(e,t,!0)),o=r.type.map(e=>l(e)).join(" or ")):(o=l(r),i="Array"===o?v(t):"Object"===o?c(t):"String"===o||"Number"===o||"Boolean"===o||"Function"===o?function(e){if(null==e)return "";const t=e.constructor.toString().match(u);return t?t[1]:""}(t)===o:t instanceof r.type);}if(!i){const e=`${a}value "${t}" should be of type "${o}"`;return !1===n?(f(e),!1):e}if(p(r,"validator")&&h(r.validator)){const e=f,o=[];if(f=e=>{o.push(e);},i=r.validator(t),f=e,!i){const e=(o.length>1?"* ":"")+o.join("\n* ");return o.length=0,!1===n?(f(e),i):e}}return i}function j(e,t){const n=Object.defineProperties(t,{_vueTypes_name:{value:e,writable:!0},isRequired:{get(){return this.required=!0,this}},def:{value(e){return void 0===e?this.type===Boolean||Array.isArray(this.type)&&this.type.includes(Boolean)?void(this.default=void 0):(p(this,"default")&&delete this.default,this):h(e)||!0===m(this,e,!0)?(this.default=v(e)?()=>[...e]:c(e)?()=>Object.assign({},e):e,this):(f(`${this._vueTypes_name} - invalid default value: "${e}"`),this)}}}),{validator:r}=n;return h(r)&&(n.validator=O(r,n)),n}function _(e,t){const n=j(e,t);return Object.defineProperty(n,"validate",{value(e){return h(this.validator)&&f(`${this._vueTypes_name} - calling .validate() will overwrite the current custom validator function. Validator info:\n${JSON.stringify(this)}`),this.validator=O(e,this),this}})}function T(e,t,r){const o=function(e){const t={};return Object.getOwnPropertyNames(e).forEach(n=>{t[n]=Object.getOwnPropertyDescriptor(e,n);}),Object.defineProperties({},t)}(t);if(o._vueTypes_name=e,!c(r))return o;const{validator:a}=r,s=n(r,i);if(h(a)){let{validator:e}=o;e&&(e=null!==(l=(u=e).__original)&&void 0!==l?l:u),o.validator=O(e?function(t){return e.call(this,t)&&a.call(this,t)}:a,o);}var u,l;return Object.assign(o,s)}function $(e){return e.replace(/^(?!\s*$)/gm,"  ")}const w=()=>_("any",{}),x=()=>_("function",{type:Function}),P=()=>_("boolean",{type:Boolean}),A=()=>_("string",{type:String}),E=()=>_("number",{type:Number}),S=()=>_("array",{type:Array}),N=()=>_("object",{type:Object}),V=()=>j("integer",{type:Number,validator(e){const t=y(e);return !1===t&&f(`integer - "${e}" is not an integer`),t}}),q=()=>j("symbol",{validator(e){const t="symbol"==typeof e;return !1===t&&f(`symbol - invalid value "${e}"`),t}}),k=()=>Object.defineProperty({type:null,validator(e){const t=null===e;return !1===t&&f("nullable - value should be null"),t}},"_vueTypes_name",{value:"nullable"});function D(e,t="custom validation failed"){if("function"!=typeof e)throw new TypeError("[VueTypes error]: You must provide a function as argument");return j(e.name||"<<anonymous function>>",{type:null,validator(n){const r=e(n);return r||f(`${this._vueTypes_name} - ${t}`),r}})}function L(e){if(!v(e))throw new TypeError("[VueTypes error]: You must provide an array as argument.");const t=`oneOf - value should be one of "${e.map(e=>"symbol"==typeof e?e.toString():e).join('", "')}".`,n={validator(n){const r=-1!==e.indexOf(n);return r||f(t),r}};if(-1===e.indexOf(null)){const t=e.reduce((e,t)=>{if(null!=t){const n=t.constructor;-1===e.indexOf(n)&&e.push(n);}return e},[]);t.length>0&&(n.type=t);}return j("oneOf",n)}function B(e){if(!v(e))throw new TypeError("[VueTypes error]: You must provide an array as argument");let t=!1,n=!1,r=[];for(let i=0;i<e.length;i+=1){const o=e[i];if(g(o)){if(h(o.validator)&&(t=!0),b(o,"oneOf")&&o.type){r=r.concat(o.type);continue}if(b(o,"nullable")){n=!0;continue}if(!0===o.type||!o.type){f('oneOfType - invalid usage of "true" and "null" as types.');continue}r=r.concat(o.type);}else r.push(o);}r=r.filter((e,t)=>r.indexOf(e)===t);const i=!1===n&&r.length>0?r:null;return j("oneOfType",t?{type:i,validator(t){const n=[],r=e.some(e=>{const r=m(e,t,!0);return "string"==typeof r&&n.push(r),!0===r});return r||f(`oneOfType - provided value does not match any of the ${n.length} passed-in validators:\n${$(n.join("\n"))}`),r}}:{type:i})}function F(e){return j("arrayOf",{type:Array,validator(t){let n="";const r=t.every(t=>(n=m(e,t,!0),!0===n));return r||f(`arrayOf - value validation error:\n${$(n)}`),r}})}function Y(e){return j("instanceOf",{type:e})}function I(e){return j("objectOf",{type:Object,validator(t){let n="";const r=Object.keys(t).every(r=>(n=m(e,t[r],!0),!0===n));return r||f(`objectOf - value validation error:\n${$(n)}`),r}})}function J(e){const t=Object.keys(e),n=t.filter(t=>{var n;return !(null===(n=e[t])||void 0===n||!n.required)}),r=j("shape",{type:Object,validator(r){if(!c(r))return !1;const i=Object.keys(r);if(n.length>0&&n.some(e=>-1===i.indexOf(e))){const e=n.filter(e=>-1===i.indexOf(e));return f(1===e.length?`shape - required property "${e[0]}" is not defined.`:`shape - required properties "${e.join('", "')}" are not defined.`),!1}return i.every(n=>{if(-1===t.indexOf(n))return !0===this._vueTypes_isLoose||(f(`shape - shape definition does not include a "${n}" property. Allowed keys: "${t.join('", "')}".`),!1);const i=m(e[n],r[n],!0);return "string"==typeof i&&f(`shape - "${n}" property validation error:\n ${$(i)}`),!0===i})}});return Object.defineProperty(r,"_vueTypes_isLoose",{writable:!0,value:!1}),Object.defineProperty(r,"loose",{get(){return this._vueTypes_isLoose=!0,this}}),r}const M=["name","validate","getter"],R=/*#__PURE__*/(()=>{var e;return (e=class{static get any(){return w()}static get func(){return x().def(this.defaults.func)}static get bool(){return void 0===this.defaults.bool?P():P().def(this.defaults.bool)}static get string(){return A().def(this.defaults.string)}static get number(){return E().def(this.defaults.number)}static get array(){return S().def(this.defaults.array)}static get object(){return N().def(this.defaults.object)}static get integer(){return V().def(this.defaults.integer)}static get symbol(){return q()}static get nullable(){return k()}static extend(e){if(f("VueTypes.extend is deprecated. Use the ES6+ method instead. See https://dwightjack.github.io/vue-types/advanced/extending-vue-types.html#extending-namespaced-validators-in-es6 for details."),v(e))return e.forEach(e=>this.extend(e)),this;const{name:t,validate:r=!1,getter:i=!1}=e,o=n(e,M);if(p(this,t))throw new TypeError(`[VueTypes error]: Type "${t}" already defined`);const{type:a}=o;if(b(a))return delete o.type,Object.defineProperty(this,t,i?{get:()=>T(t,a,o)}:{value(...e){const n=T(t,a,o);return n.validator&&(n.validator=n.validator.bind(n,...e)),n}});let s;return s=i?{get(){const e=Object.assign({},o);return r?_(t,e):j(t,e)},enumerable:!0}:{value(...e){const n=Object.assign({},o);let i;return i=r?_(t,n):j(t,n),n.validator&&(i.validator=n.validator.bind(i,...e)),i},enumerable:!0},Object.defineProperty(this,t,s)}}).defaults={},e.sensibleDefaults=void 0,e.config=r,e.custom=D,e.oneOf=L,e.instanceOf=Y,e.oneOfType=B,e.arrayOf=F,e.objectOf=I,e.shape=J,e.utils={validate:(e,t)=>!0===m(t,e,!0),toType:(e,t,n=!1)=>n?_(e,t):j(e,t)},e})();function U(e={func:()=>{},bool:!0,string:"",number:0,array:()=>[],object:()=>({}),integer:0}){var n;return (n=class extends R{static get sensibleDefaults(){return t({},this.defaults)}static set sensibleDefaults(n){this.defaults=!1!==n?t({},!0!==n?n:e):{};}}).defaults=t({},e),n}class z extends(U()){}

const CWidgetStatsB = defineComponent({
    name: 'CWidgetStatsB',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Colors have been inverted from their default dark shade.
         */
        inverse: {
            type: Boolean,
            default: undefined,
            require: false,
        },
        progress: J({
            /**
             * Sets the color context of the progress bar to one of CoreUI’s themed colors
             *
             * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
             */
            color: Color,
            /**
             * The percent to progress the ProgressBar (out of 100).
             */
            value: {
                type: Number,
                default: 0,
            },
        }),
        /**
         * Helper text for your component. If you want to pass non-string value please use dedicated slot `<template #text>...</template>`
         */
        text: {
            type: String,
            default: undefined,
            require: false,
        },
        /**
         * Title for your component. If you want to pass non-string value please use dedicated slot `<template #title>...</template>`
         */
        title: {
            type: String,
            default: undefined,
            require: false,
        },
        /**
         * Value for your component. If you want to pass non-string or non-number value please use dedicated slot `<template #value>...</template>`
         */
        value: {
            type: [Number, String],
            default: 0,
            require: false,
        },
    },
    setup(props, { slots }) {
        return () => h$1(CCard, {
            class: [
                {
                    'text-high-emphasis-inverse': props.inverse,
                },
            ],
            color: props.color,
        }, () => h$1(CCardBody, {
            class: 'card-body',
        }, () => [
            (props.value || slots.value) &&
                h$1('div', {
                    class: 'fs-4 fw-semibold',
                }, {
                    default: () => (slots.value && slots.value()) || props.value,
                }),
            (props.title || slots.title) &&
                h$1('div', {}, {
                    default: () => (slots.title && slots.title()) || props.title,
                }),
            h$1(CProgress, {
                class: 'my-2',
                ...(props.progress && props.progress.color && { color: props.progress.color }),
                height: 4,
                ...(props.progress && props.progress.value && { value: props.progress.value }),
                white: props.inverse,
            }),
            (props.text || slots.text) &&
                h$1('small', {
                    class: [
                        props.inverse ? 'text-medium-emphasis-inverse' : 'text-medium-emphasis',
                    ],
                }, {
                    default: () => (slots.text && slots.text()) || props.text,
                }),
        ]));
    },
});

const CWidgetStatsC = defineComponent({
    name: 'CWidgetStatsC',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Colors have been inverted from their default dark shade.
         */
        inverse: {
            type: Boolean,
            default: undefined,
            require: false,
        },
        progress: J({
            /**
             * Sets the color context of the progress bar to one of CoreUI’s themed colors
             *
             * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
             */
            color: Color,
            /**
             * The percent to progress the ProgressBar (out of 100).
             */
            value: {
                type: Number,
                default: 0,
            },
        }),
        /**
         * Title for your component. If you want to pass non-string value please use dedicated slot `<template #title>...</template>`
         */
        title: {
            type: String,
            default: undefined,
            require: false,
        },
        /**
         * Value for your component. If you want to pass non-string or non-number value please use dedicated slot `<template #value>...</template>`
         */
        value: {
            type: [Number, String],
            default: 0,
            require: false,
        },
    },
    /**
     * Location for icon component.
     *
     * @slot icon
     */
    setup(props, { slots }) {
        return () => h$1(CCard, {
            class: [
                {
                    ['text-white']: props.inverse,
                },
            ],
            color: props.color,
        }, () => h$1(CCardBody, {
            class: 'card-body',
        }, () => [
            slots.icon &&
                h$1('div', {
                    class: [
                        'text-end mb-4',
                        props.inverse ? 'text-medium-emphasis-inverse' : 'text-medium-emphasis',
                    ],
                }, slots.icon && slots.icon()),
            (props.value || slots.value) &&
                h$1('div', {
                    class: 'fs-4 fw-semibold',
                }, {
                    default: () => (slots.value && slots.value()) || props.value,
                }),
            (props.title || slots.title) &&
                h$1('div', {
                    class: [
                        'text-uppercase fw-semibold small',
                        props.inverse ? 'text-medium-emphasis-inverse' : 'text-medium-emphasis',
                    ],
                }, {
                    default: () => (slots.title && slots.title()) || props.title,
                }),
            h$1(CProgress, {
                class: 'my-2',
                ...(props.progress && props.progress.color && { color: props.progress.color }),
                height: 4,
                ...(props.progress && props.progress.value && { value: props.progress.value }),
                white: props.inverse,
            }),
        ]));
    },
});

const CWidgetStatsD = defineComponent({
    name: 'CWidgetStatsD',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Values and titles for your component.
         */
        values: {
            type: Array,
            default: () => [],
            require: false,
        },
    },
    /**
     * Location for icon component, ex. `<CDropdown>`.
     *
     * @slot icon
     */
    /**
     * Location for chart component.
     *
     * @slot chart
     */
    setup(props, { slots }) {
        return () => h$1(CCard, {}, {
            default: () => [
                h$1(CCardHeader, {
                    class: [
                        'position-relative d-flex justify-content-center align-items-center',
                        {
                            [`bg-${props.color}`]: props.color,
                        },
                    ],
                }, () => [slots.icon && slots.icon(), slots.chart && slots.chart()]),
                h$1(CCardBody, {
                    class: 'row text-center',
                }, {
                    default: () => props.values &&
                        props.values.map((value, index) => [
                            index % 2 !== 0 && h$1('div', { class: 'vr' }),
                            h$1(CCol, {}, {
                                default: () => [
                                    h$1(CCol, { class: 'fs-5 fw-semibold' }, () => value.value),
                                    h$1(CCol, { class: 'text-uppercase text-medium-emphasis small' }, () => value.title),
                                ],
                            }),
                        ]),
                }),
            ],
        });
    },
});

const CWidgetStatsE = defineComponent({
    name: 'CWidgetStatsE',
    props: {
        /**
         * Title for your component. If you want to pass non-string value please use dedicated slot `<template #title>...</template>`
         */
        title: {
            type: String,
            default: undefined,
            require: false,
        },
        /**
         * Value for your component. If you want to pass non-string or non-number value please use dedicated slot `<template #value>...</template>`
         */
        value: {
            type: [Number, String],
            default: 0,
            require: false,
        },
    },
    /**
     * Location for chart component.
     *
     * @slot chart
     */
    setup(props, { slots }) {
        return () => h$1(CCard, {}, () => h$1(CCardBody, {
            class: 'text-center',
        }, () => [
            (props.title || slots.title) &&
                h$1('div', {
                    class: 'text-muted small text-uppercase font-weight-bold',
                }, {
                    default: () => (slots.title && slots.title()) || props.title,
                }),
            (props.value || slots.value) &&
                h$1('div', {
                    class: 'h2 py-3',
                }, {
                    default: () => (slots.value && slots.value()) || props.value,
                }),
            slots.chart && slots.chart(),
            slots.default && slots.default(),
        ]));
    },
});

const CWidgetStatsF = defineComponent({
    name: 'CWidgetStatsF',
    props: {
        /**
         * Sets the color context of the component to one of CoreUI’s themed colors.
         *
         * @values 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'dark', 'light'
         */
        color: Color,
        /**
         * Set padding of your component.
         */
        padding: {
            type: Boolean,
            default: true,
        },
        /**
         * Title for your component. If you want to pass non-string value please use dedicated slot `<template #title>...</template>`
         */
        title: {
            type: String,
            default: undefined,
            require: false,
        },
        /**
         * Helper text for your component. If you want to pass non-string value please use dedicated slot `<template #text>...</template>`
         */
        text: {
            type: String,
            default: undefined,
            require: false,
        },
        /**
         * Value for your component. If you want to pass non-string or non-number value please use dedicated slot `<template #value>...</template>`
         */
        value: {
            type: [Number, String],
            default: 0,
            require: false,
        },
    },
    /**
     * Location for icon component.
     *
     * @slot icon
     */
    setup(props, { slots }) {
        return () => h$1(CCard, {}, {
            default: () => [
                h$1(CCardBody, {
                    class: ['d-flex align-items-center', props.padding === false && 'p-0'],
                }, () => [
                    h$1('div', {
                        class: [
                            'me-3',
                            'text-white',
                            `bg-${props.color}`,
                            props.padding ? 'p-3' : 'p-4',
                        ],
                    }, (slots.default && slots.default()) || (slots.icon && slots.icon())),
                    h$1('div', {}, [
                        (props.value || slots.value) &&
                            h$1('div', {
                                class: [`fs-6 fw-semibold text-${props.color}`],
                            }, {
                                default: () => (slots.value && slots.value()) || props.value,
                            }),
                        (props.title || slots.title) &&
                            h$1('div', {
                                class: 'text-medium-emphasis text-uppercase fw-semibold small',
                            }, {
                                default: () => (slots.title && slots.title()) || props.title,
                            }),
                    ]),
                ]),
                slots.footer && h$1(CCardFooter, {}, () => slots.footer && slots.footer()),
            ],
        });
    },
});

const CWidgetsStatsPlugin = {
    install: (app) => {
        app.component(CWidgetStatsA.name, CWidgetStatsA);
        app.component(CWidgetStatsB.name, CWidgetStatsB);
        app.component(CWidgetStatsC.name, CWidgetStatsC);
        app.component(CWidgetStatsD.name, CWidgetStatsD);
        app.component(CWidgetStatsE.name, CWidgetStatsE);
        app.component(CWidgetStatsF.name, CWidgetStatsF);
    },
};

var Components = /*#__PURE__*/Object.freeze({
    __proto__: null,
    CAccordion: CAccordion,
    CAccordionBody: CAccordionBody,
    CAccordionButton: CAccordionButton,
    CAccordionHeader: CAccordionHeader,
    CAccordionItem: CAccordionItem,
    CAccordionPlugin: CAccordionPlugin,
    CAlert: CAlert,
    CAlertHeading: CAlertHeading,
    CAlertLink: CAlertLink,
    CAlertPlugin: CAlertPlugin,
    CAvatar: CAvatar,
    CAvatarPlugin: CAvatarPlugin,
    CBackdrop: CBackdrop,
    CBackdropPlugin: CBackdropPlugin,
    CBadge: CBadge,
    CBadgePlugin: CBadgePlugin,
    CBreadcrumb: CBreadcrumb,
    CBreadcrumbItem: CBreadcrumbItem,
    CBreadcrumbPlugin: CBreadcrumbPlugin,
    CButton: CButton,
    CButtonGroup: CButtonGroup,
    CButtonGroupPlugin: CButtonGroupPlugin,
    CButtonPlugin: CButtonPlugin,
    CButtonToolbar: CButtonToolbar,
    CCLinkPlugin: CCLinkPlugin,
    CCallout: CCallout,
    CCalloutPlugin: CCalloutPlugin,
    CCard: CCard,
    CCardBody: CCardBody,
    CCardFooter: CCardFooter,
    CCardGroup: CCardGroup,
    CCardHeader: CCardHeader,
    CCardImage: CCardImage,
    CCardImageOverlay: CCardImageOverlay,
    CCardLink: CCardLink,
    CCardPlugin: CCardPlugin,
    CCardSubtitle: CCardSubtitle,
    CCardText: CCardText,
    CCardTitle: CCardTitle,
    CCarousel: CCarousel,
    CCarouselCaption: CCarouselCaption,
    CCarouselItem: CCarouselItem,
    CCarouselPlugin: CCarouselPlugin,
    CCloseButton: CCloseButton,
    CCloseButtonPlugin: CCloseButtonPlugin,
    CCol: CCol,
    CCollapse: CCollapse,
    CCollapsePlugin: CCollapsePlugin,
    CContainer: CContainer,
    CDropdown: CDropdown,
    CDropdownDivider: CDropdownDivider,
    CDropdownHeader: CDropdownHeader,
    CDropdownItem: CDropdownItem,
    CDropdownMenu: CDropdownMenu,
    CDropdownPlugin: CDropdownPlugin,
    CDropdownToggle: CDropdownToggle,
    CFooter: CFooter,
    CFooterPlugin: CFooterPlugin,
    CForm: CForm,
    CFormCheck: CFormCheck,
    CFormFeedback: CFormFeedback,
    CFormFloating: CFormFloating,
    CFormInput: CFormInput,
    CFormLabel: CFormLabel,
    CFormPlugin: CFormPlugin,
    CFormRange: CFormRange,
    CFormSelect: CFormSelect,
    CFormSwitch: CFormSwitch,
    CFormText: CFormText,
    CFormTextarea: CFormTextarea,
    CGridPlugin: CGridPlugin,
    CHeader: CHeader,
    CHeaderBrand: CHeaderBrand,
    CHeaderDivider: CHeaderDivider,
    CHeaderNav: CHeaderNav,
    CHeaderPlugin: CHeaderPlugin,
    CHeaderText: CHeaderText,
    CHeaderToggler: CHeaderToggler,
    CImage: CImage,
    CImagePlugin: CImagePlugin,
    CInputGroup: CInputGroup,
    CInputGroupText: CInputGroupText,
    CLink: CLink,
    CListGroup: CListGroup,
    CListGroupItem: CListGroupItem,
    CListGroupPlugin: CListGroupPlugin,
    CModal: CModal,
    CModalBody: CModalBody,
    CModalFooter: CModalFooter,
    CModalHeader: CModalHeader,
    CModalPlugin: CModalPlugin,
    CModalTitle: CModalTitle,
    CNav: CNav,
    CNavGroup: CNavGroup,
    CNavGroupItems: CNavGroupItems,
    CNavItem: CNavItem,
    CNavLink: CNavLink,
    CNavPlugin: CNavPlugin,
    CNavTitle: CNavTitle,
    CNavbar: CNavbar,
    CNavbarBrand: CNavbarBrand,
    CNavbarNav: CNavbarNav,
    CNavbarPlugin: CNavbarPlugin,
    CNavbarText: CNavbarText,
    CNavbarToggler: CNavbarToggler,
    COffcanvas: COffcanvas,
    COffcanvasBody: COffcanvasBody,
    COffcanvasHeader: COffcanvasHeader,
    COffcanvasPlugin: COffcanvasPlugin,
    COffcanvasTitle: COffcanvasTitle,
    CPagination: CPagination,
    CPaginationItem: CPaginationItem,
    CPaginationPlugin: CPaginationPlugin,
    CPlaceholder: CPlaceholder,
    CPlaceholderPlugin: CPlaceholderPlugin,
    CPopover: CPopover,
    CPopoverPlugin: CPopoverPlugin,
    CProgress: CProgress,
    CProgressBar: CProgressBar,
    CProgressPlugin: CProgressPlugin,
    CRow: CRow,
    CSidebar: CSidebar,
    CSidebarBrand: CSidebarBrand,
    CSidebarFooter: CSidebarFooter,
    CSidebarHeader: CSidebarHeader,
    CSidebarNav: CSidebarNav,
    CSidebarPlugin: CSidebarPlugin,
    CSidebarToggler: CSidebarToggler,
    CSpinner: CSpinner,
    CSpinnerPlugin: CSpinnerPlugin,
    CTabContent: CTabContent,
    CTabPane: CTabPane,
    CTable: CTable,
    CTableBody: CTableBody,
    CTableCaption: CTableCaption,
    CTableDataCell: CTableDataCell,
    CTableFoot: CTableFoot,
    CTableHead: CTableHead,
    CTableHeaderCell: CTableHeaderCell,
    CTablePlugin: CTablePlugin,
    CTableRow: CTableRow,
    CTabsPlugin: CTabsPlugin,
    CToast: CToast,
    CToastBody: CToastBody,
    CToastClose: CToastClose,
    CToastHeader: CToastHeader,
    CToastPlugin: CToastPlugin,
    CToaster: CToaster,
    CTooltip: CTooltip,
    CTooltipPlugin: CTooltipPlugin,
    CWidgetStatsA: CWidgetStatsA,
    CWidgetStatsB: CWidgetStatsB,
    CWidgetStatsC: CWidgetStatsC,
    CWidgetStatsD: CWidgetStatsD,
    CWidgetStatsE: CWidgetStatsE,
    CWidgetStatsF: CWidgetStatsF,
    CWidgetsStatsPlugin: CWidgetsStatsPlugin
});

const BREAKPOINTS = [
    'xxl',
    'xl',
    'lg',
    'md',
    'sm',
    'xs',
];
var vcplaceholder = {
    name: 'c-placeholder',
    mounted(el, binding) {
        const value = binding.value;
        el.classList.add(value.animation ? `placeholder-${value.animation}` : 'placeholder');
        BREAKPOINTS.forEach((bp) => {
            const breakpoint = value[bp];
            const infix = bp === 'xs' ? '' : `-${bp}`;
            if (typeof breakpoint === 'number') {
                el.classList.add(`col${infix}-${breakpoint}`);
            }
            if (typeof breakpoint === 'boolean') {
                el.classList.add(`col${infix}`);
            }
        });
    },
};

const getUID$1 = (prefix) => {
    do {
        prefix += Math.floor(Math.random() * 1000000);
    } while (document.getElementById(prefix));
    return prefix;
};
const createPopoverElement = (id, header, content) => {
    const popover = document.createElement('div');
    popover.id = id;
    popover.classList.add('popover', 'bs-popover-auto', 'fade');
    popover.setAttribute('role', 'popover');
    popover.innerHTML = `<div class="popover-arrow" data-popper-arrow></div>
     <div class="popover-header">${header}</div>
     <div class="popover-body" id="">${content}</div>`;
    return popover;
};
const addPopoverElement = (popover, el, popperOptions) => {
    document.body.appendChild(popover);
    createPopper(el, popover, popperOptions);
    setTimeout(() => {
        popover.classList.add('show');
    }, 1);
};
const removePopoverElement = (popover) => {
    popover.classList.remove('show');
    setTimeout(() => {
        document.body.removeChild(popover);
    }, 300);
};
const togglePopoverElement = (popover, el, popperOptions) => {
    const popperElement = document.getElementById(popover.id);
    if (popperElement && popperElement.classList.contains('show')) {
        removePopoverElement(popover);
        return;
    }
    addPopoverElement(popover, el, popperOptions);
};
var vcpopover = {
    name: 'c-popover',
    uid: '',
    mounted(el, binding) {
        const value = binding.value;
        const content = typeof value === 'string' ? value : value.content ? value.content : '';
        const header = value.header ? value.header : '';
        const trigger = value.trigger ? value.trigger : 'click';
        // Popper Config
        const offset = value.offset ? value.offset : [0, 8];
        const placement = value.placement ? value.placement : 'top';
        const popperOptions = {
            placement: placement,
            modifiers: [
                {
                    name: 'offset',
                    options: {
                        offset: offset,
                    },
                },
            ],
        };
        const popoverUID = getUID$1('popover');
        binding.arg = popoverUID;
        const popover = createPopoverElement(popoverUID, header, content);
        trigger.includes('click') &&
            el.addEventListener('click', () => {
                togglePopoverElement(popover, el, popperOptions);
            });
        if (trigger.includes('focus')) {
            el.addEventListener('focus', () => {
                addPopoverElement(popover, el, popperOptions);
            });
            el.addEventListener('blur', () => {
                removePopoverElement(popover);
            });
        }
        if (trigger.includes('hover')) {
            el.addEventListener('mouseenter', () => {
                addPopoverElement(popover, el, popperOptions);
            });
            el.addEventListener('mouseleave', () => {
                removePopoverElement(popover);
            });
        }
    },
    unmounted(_el, binding) {
        const popover = binding.arg && document.getElementById(binding.arg);
        popover && popover.remove();
    },
};

const getUID = (prefix) => {
    do {
        prefix += Math.floor(Math.random() * 1000000);
    } while (document.getElementById(prefix));
    return prefix;
};
const createTooltipElement = (id, content) => {
    const tooltip = document.createElement('div');
    tooltip.id = id;
    tooltip.classList.add('tooltip', 'bs-tooltip-auto', 'fade');
    tooltip.setAttribute('role', 'tooltip');
    tooltip.innerHTML = `<div class="tooltip-arrow" data-popper-arrow></div>
     <div class="tooltip-inner" id="">${content}</div>`;
    return tooltip;
};
const addTooltipElement = (tooltip, el, popperOptions) => {
    document.body.appendChild(tooltip);
    createPopper(el, tooltip, popperOptions);
    setTimeout(() => {
        tooltip.classList.add('show');
    }, 1);
};
const removeTooltipElement = (tooltip) => {
    tooltip.classList.remove('show');
    setTimeout(() => {
        document.body.removeChild(tooltip);
    }, 300);
};
const toggleTooltipElement = (tooltip, el, popperOptions) => {
    const popperElement = document.getElementById(tooltip.id);
    if (popperElement && popperElement.classList.contains('show')) {
        removeTooltipElement(tooltip);
        return;
    }
    addTooltipElement(tooltip, el, popperOptions);
};
var vctooltip = {
    mounted(el, binding) {
        const value = binding.value;
        const content = typeof value === 'string' ? value : value.content ? value.content : '';
        const trigger = value.trigger ? value.trigger : 'hover';
        // Popper Config
        const offset = value.offset ? value.offset : [0, 0];
        const placement = value.placement ? value.placement : 'top';
        const popperOptions = {
            placement: placement,
            modifiers: [
                {
                    name: 'offset',
                    options: {
                        offset: offset,
                    },
                },
            ],
        };
        const tooltipUID = getUID('tooltip');
        binding.arg = tooltipUID;
        const tooltip = createTooltipElement(tooltipUID, content);
        trigger.includes('click') &&
            el.addEventListener('click', () => {
                toggleTooltipElement(tooltip, el, popperOptions);
            });
        if (trigger.includes('focus')) {
            el.addEventListener('focus', () => {
                addTooltipElement(tooltip, el, popperOptions);
            });
            el.addEventListener('blur', () => {
                removeTooltipElement(tooltip);
            });
        }
        if (trigger.includes('hover')) {
            el.addEventListener('mouseenter', () => {
                addTooltipElement(tooltip, el, popperOptions);
            });
            el.addEventListener('mouseleave', () => {
                removeTooltipElement(tooltip);
            });
        }
    },
    beforeUnmount(_el, binding) {
        const tooltip = binding.arg && document.getElementById(binding.arg);
        tooltip && tooltip.remove();
    },
};

const removeKeysFromObject = (object, keys) => {
    return Object.entries(object).reduce((obj, [key, value]) => {
        if (!keys.includes(key) && !keys.includes(value.name)) {
            obj[key] = value;
        }
        return obj;
    }, {});
};
const CoreuiVue = {
    install: (app, options) => {
        let pluginComponents = Components;
        // let pluginDirectives = Directives
        const toRemove = options && options.remove ? options.remove : null;
        if (toRemove && Array.isArray(toRemove)) {
            pluginComponents = removeKeysFromObject(Components, toRemove);
            // pluginDirectives = removeKeysFromObject(Directives, toRemove)
        }
        for (const plugin in pluginComponents) {
            app.component(plugin, Components[plugin]);
        }
        // for (const directive in pluginDirectives) {
        //   app.directive(directive, Directives[directive])
        // }
        app.directive('c-placeholder', vcplaceholder);
        app.directive('c-popover', vcpopover);
        app.directive('c-tooltip', vctooltip);
    },
};

export { CAccordion, CAccordionBody, CAccordionButton, CAccordionHeader, CAccordionItem, CAccordionPlugin, CAlert, CAlertHeading, CAlertLink, CAlertPlugin, CAvatar, CAvatarPlugin, CBackdrop, CBackdropPlugin, CBadge, CBadgePlugin, CBreadcrumb, CBreadcrumbItem, CBreadcrumbPlugin, CButton, CButtonGroup, CButtonGroupPlugin, CButtonPlugin, CButtonToolbar, CCLinkPlugin, CCallout, CCalloutPlugin, CCard, CCardBody, CCardFooter, CCardGroup, CCardHeader, CCardImage, CCardImageOverlay, CCardLink, CCardPlugin, CCardSubtitle, CCardText, CCardTitle, CCarousel, CCarouselCaption, CCarouselItem, CCarouselPlugin, CCloseButton, CCloseButtonPlugin, CCol, CCollapse, CCollapsePlugin, CContainer, CDropdown, CDropdownDivider, CDropdownHeader, CDropdownItem, CDropdownMenu, CDropdownPlugin, CDropdownToggle, CFooter, CFooterPlugin, CForm, CFormCheck, CFormFeedback, CFormFloating, CFormInput, CFormLabel, CFormPlugin, CFormRange, CFormSelect, CFormSwitch, CFormText, CFormTextarea, CGridPlugin, CHeader, CHeaderBrand, CHeaderDivider, CHeaderNav, CHeaderPlugin, CHeaderText, CHeaderToggler, CImage, CImagePlugin, CInputGroup, CInputGroupText, CLink, CListGroup, CListGroupItem, CListGroupPlugin, CModal, CModalBody, CModalFooter, CModalHeader, CModalPlugin, CModalTitle, CNav, CNavGroup, CNavGroupItems, CNavItem, CNavLink, CNavPlugin, CNavTitle, CNavbar, CNavbarBrand, CNavbarNav, CNavbarPlugin, CNavbarText, CNavbarToggler, COffcanvas, COffcanvasBody, COffcanvasHeader, COffcanvasPlugin, COffcanvasTitle, CPagination, CPaginationItem, CPaginationPlugin, CPlaceholder, CPlaceholderPlugin, CPopover, CPopoverPlugin, CProgress, CProgressBar, CProgressPlugin, CRow, CSidebar, CSidebarBrand, CSidebarFooter, CSidebarHeader, CSidebarNav, CSidebarPlugin, CSidebarToggler, CSpinner, CSpinnerPlugin, CTabContent, CTabPane, CTable, CTableBody, CTableCaption, CTableDataCell, CTableFoot, CTableHead, CTableHeaderCell, CTablePlugin, CTableRow, CTabsPlugin, CToast, CToastBody, CToastClose, CToastHeader, CToastPlugin, CToaster, CTooltip, CTooltipPlugin, CWidgetStatsA, CWidgetStatsB, CWidgetStatsC, CWidgetStatsD, CWidgetStatsE, CWidgetStatsF, CWidgetsStatsPlugin, CoreuiVue as default, vcplaceholder, vcpopover, vctooltip };
//# sourceMappingURL=index.es.js.map
